<?php
// Zaman aşımını artıralım (büyük dosyalar için)
set_time_limit(300); // 5 dakika

// Gerekli dosyaları dahil et
require_once __DIR__ . '/../config/db_connect.php';

// Oturum kontrolü
if (!isset($_SESSION['admin_user_id'])) {
    header('HTTP/1.1 403 Forbidden');
    exit('Erişim engellendi.');
}

// --- Lisans ID'sini Al ve Doğrula ---
if (!isset($_GET['license_id']) || !filter_var($_GET['license_id'], FILTER_VALIDATE_INT)) {
    die('Geçersiz lisans ID\'si.');
}
$license_id = (int)$_GET['license_id'];

// --- Gerekli Bilgileri Veritabanından Çek ---
$sql = "SELECT
            l.license_key, l.domain,
            p.name AS product_name, p.original_zip_path, p.critical_files, p.injection_placeholder
        FROM licenses l
        JOIN products p ON l.product_id = p.id
        WHERE l.id = ?";
$stmt = $conn->prepare($sql);
if (!$stmt) { error_log("Package Error - SQL prepare error: " . $conn->error); die('Veritabanı sorgu hatası (prepare).'); }
$stmt->bind_param("i", $license_id);
$stmt->execute();
$result = $stmt->get_result();
if ($result->num_rows !== 1) { die('Lisans bulunamadı (ID: ' . $license_id . ').'); }
$data = $result->fetch_assoc();
$stmt->close();
$conn->close();

// --- Gerekli Verileri Kontrol Et ---
if (empty($data['original_zip_path'])) { die('Hata: Bu ürün için orijinal ZIP dosyası yolu tanımlanmamış.'); }
if (empty($data['injection_placeholder'])) { die('Hata: Bu ürün için enjeksiyon yer tutucusu tanımlanmamış.'); }
if (empty($data['license_key'])) { die('Hata: Bu lisans için bir lisans anahtarı bulunamadı.'); }

$original_zip_full_path = realpath(__DIR__ . '/../' . $data['original_zip_path']);
if ($original_zip_full_path === false || !file_exists($original_zip_full_path)) {
     error_log("Package Error - Original ZIP not found at expected path: " . __DIR__ . '/../' . $data['original_zip_path']);
    die('Orijinal ürün ZIP dosyası sunucuda bulunamadı: ' . htmlspecialchars($data['original_zip_path']));
}
error_log("Package Info - Using original ZIP: " . $original_zip_full_path);

// Kritik dosyaları diziye çevir (eğer boş değilse)
$critical_files_array = [];
if (!empty($data['critical_files'])) {
    $critical_files_array = array_map('trim', explode(',', $data['critical_files']));
    if (count($critical_files_array) === 1 && $critical_files_array[0] === '') { $critical_files_array = []; }
}
if (!empty($critical_files_array)) { error_log("Package Info - Critical files specified: " . implode(', ', $critical_files_array)); }
else { error_log("Package Info - No critical files specified, will scan all PHP files."); }

// --- Geçici Klasör ve Ayarlar ---
$temp_extract_dir = rtrim(sys_get_temp_dir(), DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . 'license_pkg_' . uniqid() . DIRECTORY_SEPARATOR;
$output_zip_filename = preg_replace('/[^a-zA-Z0-9\-_\.]/', '_', $data['product_name'])
                     . '_' . preg_replace('/[^a-zA-Z0-9\-_\.]/', '_', $data['domain'] ?: 'all')
                     . '_licensed.zip';
error_log("Package Info - Temp extract dir: " . $temp_extract_dir);

// --- Basit Obfuscation Fonksiyonu (Sadece Base64, PHP Etiketleri OLMADAN) ---
function simple_obfuscate($code)
{
    $code_body = trim(preg_replace(['/^\s*<\?php\s*/i', '/\s*\?>\s*$/'], '', $code));
    $encoded = base64_encode($code_body);
    // Tek satırlık yorum kullanalım
    $obfuscated_code = "// Encoded License Check\neval(base64_decode('" . $encoded . "'));\n// End Check";
    return $obfuscated_code;
}

// --- Enjekte Edilecek Lisans Kontrol Kodu (Oturum Bazlı, Otomatik İlk Çağrı - PHP Etiketleri OLMADAN) ---
$base_url_for_api = rtrim(BASE_URL, '/');
$api_endpoint_url = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https" : "http")
                 . "://{$_SERVER['HTTP_HOST']}" . $base_url_for_api . '/api/validate.php';
error_log("Package Info - API Endpoint URL: " . $api_endpoint_url);

// Kodu PHP etiketleri OLMADAN tanımla
$code_to_inject_raw = <<<PHP
// --- License Check Function and Initial Call - Do Not Modify ---
if (!function_exists('__verify_license')) {
    function __verify_license(\$checkIntervalSeconds = 3600) {
        if (session_status() == PHP_SESSION_NONE) { @session_start(); }
        \$sessionKey = 'lc_status_' . substr(md5('{$data['license_key']}'), 0, 8);
        \$lastCheckKey = \$sessionKey . '_ts';
        if (isset(\$_SESSION[\$sessionKey]) && \$_SESSION[\$sessionKey] === true && isset(\$_SESSION[\$lastCheckKey]) && (time() - \$_SESSION[\$lastCheckKey] < \$checkIntervalSeconds)) { return true; }
        \$lk='{$data['license_key']}';\$ae='{$api_endpoint_url}';\$cd=\$_SERVER['HTTP_HOST']??php_uname('n');\$ld=['license_key'=>\$lk,'domain'=>\$cd];\$lu=\$ae.'?'.http_build_query(\$ld);
        \$lo=['http'=>['method'=>'GET','timeout'=>10,'ignore_errors'=>true]];\$lc=stream_context_create(\$lo);\$lrj=@file_get_contents(\$lu,false,\$lc);
        if(\$lrj===false){ unset(\$_SESSION[\$sessionKey], \$_SESSION[\$lastCheckKey]); return false; }
        \$lr=json_decode(\$lrj,true);
        if(\$lr===null||!isset(\$lr['status'])||\$lr['status']!=='valid'){ unset(\$_SESSION[\$sessionKey], \$_SESSION[\$lastCheckKey]); return false; }
        \$_SESSION[\$sessionKey] = true; \$_SESSION[\$lastCheckKey] = time(); unset(\$lk,\$ae,\$cd,\$ld,\$lu,\$lo,\$lc,\$lrj,\$lr); return true;
    }
    if (!__verify_license(true)) { header('HTTP/1.1 403 Forbidden'); die("License Validation Failed."); }
}
// --- End License Check ---
PHP; // Heredoc sonu

// Obfuscation tekrar aktif edildi
$code_to_inject = simple_obfuscate($code_to_inject_raw);

// --- ZIP Açma ---
$zip = new ZipArchive;
if (!@mkdir($temp_extract_dir, 0777, true)) { $last_error = error_get_last(); error_log("Package Error - Could not create temp directory: " . $temp_extract_dir . " - Error: " . ($last_error['message'] ?? 'Unknown')); die('Geçici klasör oluşturulamadı. İzinleri kontrol edin.'); }
if ($zip->open($original_zip_full_path) !== TRUE) { @rmdir($temp_extract_dir); error_log("Package Error - Could not open original ZIP: " . $original_zip_full_path); die('Orijinal ZIP dosyası açılamadı.'); }
if (!$zip->extractTo($temp_extract_dir)) { $zip->close(); error_log("Package Error - Could not extract ZIP content to: " . $temp_extract_dir); recursive_delete($temp_extract_dir); die('ZIP içeriği çıkarılamadı.'); }
$zip->close();
error_log("Package Info - ZIP extracted successfully to: " . $temp_extract_dir);

// --- Kod Enjeksiyonu ---
$placeholder = $data['injection_placeholder'];
$files_processed = 0;
$placeholder_found_in_any_file = false;

if (!empty($critical_files_array)) {
    // Mod 1: Belirtilen Kritik Dosyaları İşle
    error_log("Package Info - Processing specified critical files: " . implode(', ', $critical_files_array));
    foreach ($critical_files_array as $relative_path) {
        $file_full_path = $temp_extract_dir . ltrim($relative_path, '/\\');
        if (file_exists($file_full_path) && is_readable($file_full_path) && is_writable($file_full_path)) {
            $content = file_get_contents($file_full_path);
            if (strpos($content, $placeholder) !== false) {
                $placeholder_found_in_any_file = true;
                error_log("Package Info - Placeholder found in critical file: " . $file_full_path);
                $new_content = str_replace($placeholder, $code_to_inject, $content, $count); // Gizlenmiş kodu kullan
                if ($count > 0 && file_put_contents($file_full_path, $new_content) !== false) { $files_processed++; error_log("Package Info - Successfully injected code into: " . $file_full_path); }
                else { error_log("Package Error - Could not write to critical file (or placeholder not replaced): " . $file_full_path); }
            } else { error_log("Package Warning - Placeholder not found in specified critical file: " . $file_full_path); }
        } else { error_log("Package Error - Specified critical file not found or not accessible: " . $file_full_path); }
    }
} else {
    // Mod 2: Tüm .php Dosyalarını Tara
    error_log("Package Info - Critical files list is empty, scanning all .php files.");
    $iterator = new RecursiveIteratorIterator( new RecursiveDirectoryIterator($temp_extract_dir, RecursiveDirectoryIterator::SKIP_DOTS | RecursiveDirectoryIterator::FOLLOW_SYMLINKS), RecursiveIteratorIterator::SELF_FIRST);
    foreach ($iterator as $file) {
        if ($file->isFile() && strtolower($file->getExtension()) === 'php' && $file->isReadable() && $file->isWritable()) {
            $file_full_path = $file->getRealPath();
            $content = file_get_contents($file_full_path);
            if (strpos($content, $placeholder) !== false) {
                $placeholder_found_in_any_file = true;
                error_log("Package Info - Placeholder found in file: " . $file_full_path);
                $new_content = str_replace($placeholder, $code_to_inject, $content, $count); // Gizlenmiş kodu kullan
                if ($count > 0 && file_put_contents($file_full_path, $new_content) !== false) { $files_processed++; error_log("Package Info - Successfully injected code into: " . $file_full_path); }
                else { error_log("Package Error - Could not write to file (or placeholder not replaced): " . $file_full_path); }
            }
        }
    }
}

// --- Temizlik Fonksiyonu ---
function recursive_delete($dir) { if (!is_dir($dir)) return; $files = array_diff(scandir($dir), array('.', '..')); foreach ($files as $file) { $path = $dir . DIRECTORY_SEPARATOR . $file; is_dir($path) ? recursive_delete($path) : @unlink($path); } @rmdir($dir); }

// --- İşlem Sonucu Kontrolü ---
if (!$placeholder_found_in_any_file) { error_log("Package Error - Placeholder not found in any processed file."); recursive_delete($temp_extract_dir); die('Hata: Belirtilen yer tutucu (' . htmlspecialchars($placeholder) . ') işlenen hiçbir dosyada bulunamadı.'); }
elseif ($files_processed === 0) { error_log("Package Error - Placeholder found but could not inject into any file (permissions?)."); recursive_delete($temp_extract_dir); die('Hata: Yer tutucu bulundu ancak kod hiçbir dosyaya yazılamadı. Dosya/klasör izinlerini kontrol edin.'); }
error_log("Package Info - Code injected into " . $files_processed . " file(s).");

// --- Yeniden ZIP'leme ---
$new_zip = new ZipArchive();
$temp_output_zip_path = rtrim(sys_get_temp_dir(), DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . $output_zip_filename;
if ($new_zip->open($temp_output_zip_path, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== TRUE) { error_log("Package Error - Could not create output ZIP file: " . $temp_output_zip_path); recursive_delete($temp_extract_dir); die('Lisanslı ZIP dosyası oluşturulamadı.'); }
$files_to_zip = new RecursiveIteratorIterator( new RecursiveDirectoryIterator($temp_extract_dir, RecursiveDirectoryIterator::SKIP_DOTS), RecursiveIteratorIterator::LEAVES_ONLY);
foreach ($files_to_zip as $name => $file) { if (!$file->isDir()) { $filePath = $file->getRealPath(); $relativePath = substr($filePath, strlen($temp_extract_dir)); if (!$new_zip->addFile($filePath, $relativePath)) { error_log("Package Error - Could not add file to ZIP: " . $filePath); } } }
$zip_status = $new_zip->close();
if ($zip_status !== true) { error_log("Package Error - Could not close output ZIP file. Status code: " . $zip_status); recursive_delete($temp_extract_dir); @unlink($temp_output_zip_path); die('Lisanslı ZIP dosyası kapatılamadı.'); }
error_log("Package Info - Output ZIP created: " . $temp_output_zip_path);

// Geçici çıkarma klasörünü sil
recursive_delete($temp_extract_dir);

// --- İndirme Başlıklarını Gönder ---
if (file_exists($temp_output_zip_path)) {
    header('Content-Description: File Transfer'); header('Content-Type: application/zip'); header('Content-Disposition: attachment; filename="' . basename($output_zip_filename) . '"');
    header('Expires: 0'); header('Cache-Control: must-revalidate'); header('Pragma: public'); header('Content-Length: ' . filesize($temp_output_zip_path));
    if (ob_get_level()) { ob_end_clean(); }
    readfile($temp_output_zip_path);
    @unlink($temp_output_zip_path);
    exit;
} else { error_log("Package Error - Final ZIP file not found after creation: " . $temp_output_zip_path); die('Oluşturulan ZIP dosyası bulunamadı.'); }

?>
