<?php
// Gerekli dosyaları dahil et
require_once __DIR__ . '/../config/db_connect.php'; // db_connect.php, config.php'yi zaten dahil ediyor

// Yanıtı JSON olarak ayarlayalım
header('Content-Type: application/json');

// --- Yardımcı Fonksiyonlar ---

/**
 * API yanıtını loglar ve JSON olarak gönderir.
 * @param mysqli $db_conn Veritabanı bağlantısı
 * @param string $status 'valid' veya 'invalid'
 * @param string|null $message İsteğe bağlı mesaj
 * @param array $log_data Loglanacak ek veriler (key, domain, reason vb.)
 * @param array $extra_response_data Yanıta eklenecek ek veriler
 */
function log_and_send_response($db_conn, $status, $message = null, $log_data = [], $extra_response_data = []) {
    // Log verilerini hazırla
    $ip_address = $_SERVER['REMOTE_ADDR'] ?? null;
    $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? null;
    $license_key_sent = $log_data['license_key'] ?? null;
    $domain_sent = $log_data['domain'] ?? null;
    $reason = $log_data['reason'] ?? null;

    // Veritabanına log kaydı yap
    $log_sql = "INSERT INTO api_logs (license_key_sent, domain_sent, ip_address, user_agent, response_status, response_reason, response_message)
                VALUES (?, ?, ?, ?, ?, ?, ?)";
    $log_stmt = $db_conn->prepare($log_sql);
    if ($log_stmt) {
        $log_stmt->bind_param("sssssss",
            $license_key_sent,
            $domain_sent,
            $ip_address,
            $user_agent,
            $status, // valid veya invalid
            $reason,
            $message
        );
        if (!$log_stmt->execute()) {
            // Loglama hatasını PHP günlüğüne yaz, ama işlemi durdurma
            error_log("API Loglama Hatası: " . $log_stmt->error);
        }
        $log_stmt->close();
    } else {
        error_log("API Loglama SQL Prepare Hatası: " . $db_conn->error);
    }

    // JSON yanıtını oluştur
    $response = ['status' => $status];
    if ($message !== null) {
        $response['message'] = $message;
    }
    // Yanıta 'reason' ekleyelim (hata durumunda)
    if ($status === 'invalid' && $reason !== null) {
         $response['reason'] = $reason;
    }
    if (!empty($extra_response_data)) {
        $response = array_merge($response, $extra_response_data);
    }

    // Yanıtı gönder ve script'i sonlandır
    echo json_encode($response);
    exit;
}

// --- Girdi Alımı ve Doğrulama ---
$license_key = trim($_REQUEST['license_key'] ?? '');
$request_domain = trim($_REQUEST['domain'] ?? '');

// Log verileri için dizi
$current_log_data = [
    'license_key' => $license_key,
    'domain' => $request_domain,
    'reason' => null // Başlangıçta null
];

// Gerekli parametreler eksik mi?
if (empty($license_key)) {
    $current_log_data['reason'] = 'missing_key';
    log_and_send_response($conn, 'invalid', 'License key is required.', $current_log_data);
}
if (empty($request_domain)) {
    $current_log_data['reason'] = 'missing_domain';
    log_and_send_response($conn, 'invalid', 'Domain is required.', $current_log_data);
}

// --- Veritabanı Sorgusu ---
$sql = "SELECT id, product_id, domain, status, expires_at
        FROM licenses
        WHERE license_key = ?
        LIMIT 1";
$stmt = $conn->prepare($sql);

if ($stmt === false) {
    error_log("API Validate - SQL prepare hatası: " . $conn->error);
    $current_log_data['reason'] = 'db_prepare_error';
    log_and_send_response($conn, 'invalid', 'Internal server error during license check.', $current_log_data);
}

$stmt->bind_param("s", $license_key);
$stmt->execute();
$result = $stmt->get_result();

// --- Doğrulama Mantığı ---
if ($result->num_rows === 1) {
    $license = $result->fetch_assoc();

    // 1. Durum Kontrolü
    if ($license['status'] !== 'active') {
        $current_log_data['reason'] = 'status_' . $license['status']; // örn: status_inactive
        log_and_send_response($conn, 'invalid', 'License is not active.', $current_log_data);
    }

    // 2. Son Kullanma Tarihi Kontrolü
    if ($license['expires_at'] !== null) {
        try {
            $expiry_date = new DateTime($license['expires_at']);
            $now = new DateTime();
            if ($now > $expiry_date) {
                $current_log_data['reason'] = 'expired';
                log_and_send_response($conn, 'invalid', 'License has expired.', $current_log_data);
            }
        } catch (Exception $e) {
            error_log("API Validate - Geçersiz son kullanma tarihi formatı: " . $license['expires_at'] . " - Hata: " . $e->getMessage());
            $current_log_data['reason'] = 'invalid_date_format';
            log_and_send_response($conn, 'invalid', 'Invalid license data (expiration).', $current_log_data);
        }
    }

    // 3. Domain Kontrolü
    $allowed_domain = $license['domain'];
    if ($allowed_domain !== null && $allowed_domain !== '*' && !empty($allowed_domain)) {
        $request_domain_normalized = preg_replace('/^www\./i', '', $request_domain);
        $allowed_domain_normalized = preg_replace('/^www\./i', '', $allowed_domain);

        if (strcasecmp($request_domain_normalized, $allowed_domain_normalized) !== 0) {
            $current_log_data['reason'] = 'domain_mismatch';
            log_and_send_response($conn, 'invalid', 'License is not valid for this domain.', $current_log_data);
        }
    }

    // --- Başarılı Doğrulama ---
    $current_log_data['reason'] = 'valid'; // Log için geçerli olduğunu belirtelim
    log_and_send_response($conn, 'valid', 'License is valid.', $current_log_data);

} else {
    // Lisans anahtarı bulunamadı
    $current_log_data['reason'] = 'not_found';
    log_and_send_response($conn, 'invalid', 'License key not found.', $current_log_data);
}

// Statement ve bağlantıyı kapat (Normalde buraya ulaşılmaz)
$stmt->close();
$conn->close();

?>
