<?php
/**
 * Dijital Menü - Menü Özellikleri Sayfası
 */

// İstanbul saatini ayarla
date_default_timezone_set('Europe/Istanbul');

// Gerekli dosyaları dahil et
require_once '../includes/auth.php';
require_once '../includes/language.php';

// Admin paneli dilini al
$adminLanguage = getAdminLanguage();

// Giriş yapılmamışsa yönlendir
redirectIfNotLoggedIn('login.php');

// Admin yetkisi kontrolü
requirePermission('admin', 'login.php');

// Sayfa başlığı
$pageTitle = __('admin.item_features', [], $adminLanguage);

// Ayarları al
$settings = getSetting();

// Özellik işlemleri
$action = isset($_GET['action']) ? $_GET['action'] : 'list';
$featureId = isset($_GET['id']) ? intval($_GET['id']) : 0;
$error = '';
$success = '';

// Veritabanı tablosunu oluştur (eğer yoksa)
$conn->query("CREATE TABLE IF NOT EXISTS features (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(50) NOT NULL,
    icon VARCHAR(255),
    color VARCHAR(20) DEFAULT '#3490dc',
    description TEXT,
    sort_order INT DEFAULT 0,
    status TINYINT(1) DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB");

// Menü öğeleri ve özellikler ilişki tablosunu oluştur (eğer yoksa)
$conn->query("CREATE TABLE IF NOT EXISTS item_features (
    item_id INT NOT NULL,
    feature_id INT NOT NULL,
    PRIMARY KEY (item_id, feature_id),
    FOREIGN KEY (item_id) REFERENCES items(id) ON DELETE CASCADE,
    FOREIGN KEY (feature_id) REFERENCES features(id) ON DELETE CASCADE
) ENGINE=InnoDB");

// Varsayılan özellikleri ekle (eğer yoksa)
$result = $conn->query("SELECT COUNT(*) as count FROM features");
$row = $result->fetch_assoc();

if ($row['count'] == 0) {
    // Varsayılan özellikleri ekle
    $conn->query("INSERT INTO features (name, description, icon, color, sort_order, status) VALUES 
    ('Özel Ürün', 'Şefin özel tavsiyesi', 'fa-star', '#dc3545', 1, 1),
    ('Popüler Ürün', 'En çok tercih edilen ürünler', 'fa-fire', '#28a745', 2, 1),
    ('Vejetaryen', 'Et içermeyen ürünler', 'fa-leaf', '#17a2b8', 3, 1),
    ('Vegan', 'Hayvansal ürün içermeyen ürünler', 'fa-seedling', '#007bff', 4, 1),
    ('Glutensiz', 'Gluten içermeyen ürünler', 'fa-wheat-awn-circle-exclamation', '#ffc107', 5, 1)");
    
    // Mevcut menü öğelerinin özelliklerini ilişki tablosuna ekle
    $conn->query("INSERT INTO item_features (item_id, feature_id)
    SELECT id, 1 FROM items WHERE is_special = 1");
    
    $conn->query("INSERT INTO item_features (item_id, feature_id)
    SELECT id, 2 FROM items WHERE is_popular = 1");
    
    $conn->query("INSERT INTO item_features (item_id, feature_id)
    SELECT id, 3 FROM items WHERE is_vegetarian = 1");
    
    $conn->query("INSERT INTO item_features (item_id, feature_id)
    SELECT id, 4 FROM items WHERE is_vegan = 1");
    
    $conn->query("INSERT INTO item_features (item_id, feature_id)
    SELECT id, 5 FROM items WHERE is_gluten_free = 1");
}

// Özellik silme işlemi
if (isset($_GET['delete']) && $featureId > 0) {
    // Özelliği sil
    $stmt = $conn->prepare("DELETE FROM features WHERE id = ?");
    $stmt->bind_param("i", $featureId);
    
    if ($stmt->execute()) {
        // İlişkileri de sil
        $conn->query("DELETE FROM item_features WHERE feature_id = $featureId");
        
        $_SESSION['success_message'] = 'Özellik başarıyla silindi.';
    } else {
        $_SESSION['error_message'] = 'Özellik silinirken bir hata oluştu.';
    }
    
    redirect('features.php');
}

// Özellik ekleme/düzenleme işlemi
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Form verilerini al
    $name = isset($_POST['name']) ? clean($_POST['name']) : '';
    $description = isset($_POST['description']) ? clean($_POST['description']) : '';
    $icon = isset($_POST['icon']) ? clean($_POST['icon']) : '';
    $color = isset($_POST['color']) ? clean($_POST['color']) : '#3490dc';
    $sortOrder = isset($_POST['sort_order']) ? intval($_POST['sort_order']) : 0;
    $status = isset($_POST['status']) ? 1 : 0;
    
    // Boş alan kontrolü
    if (empty($name)) {
        $error = 'Özellik adı boş bırakılamaz.';
    } else {
        if ($action === 'add') {
            // Özellik ekle
            $stmt = $conn->prepare("INSERT INTO features (name, description, icon, color, sort_order, status) VALUES (?, ?, ?, ?, ?, ?)");
            $stmt->bind_param("ssssii", $name, $description, $icon, $color, $sortOrder, $status);
            
            if ($stmt->execute()) {
                $_SESSION['success_message'] = 'Özellik başarıyla eklendi.';
                redirect('features.php');
            } else {
                $error = 'Özellik eklenirken bir hata oluştu.';
            }
        } elseif ($action === 'edit' && $featureId > 0) {
            // Özellik güncelle
            $stmt = $conn->prepare("UPDATE features SET name = ?, description = ?, icon = ?, color = ?, sort_order = ?, status = ? WHERE id = ?");
            $stmt->bind_param("ssssiii", $name, $description, $icon, $color, $sortOrder, $status, $featureId);
            
            if ($stmt->execute()) {
                $_SESSION['success_message'] = 'Özellik başarıyla güncellendi.';
                redirect('features.php');
            } else {
                $error = 'Özellik güncellenirken bir hata oluştu.';
            }
        }
    }
}

// Özellik bilgilerini al
$feature = [];
if ($action === 'edit' && $featureId > 0) {
    $stmt = $conn->prepare("SELECT * FROM features WHERE id = ?");
    $stmt->bind_param("i", $featureId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows === 1) {
        $feature = $result->fetch_assoc();
    } else {
        $_SESSION['error_message'] = 'Özellik bulunamadı.';
        redirect('features.php');
    }
}

// Arama ve filtreleme
$search = isset($_GET['search']) ? clean($_GET['search']) : '';
$statusFilter = isset($_GET['status']) ? $_GET['status'] : '';

// Özellikleri al - features tablosundan veri çekelim
$sql = "SELECT * FROM features";
$conditions = [];

// Arama filtresi
if (!empty($search)) {
    $conditions[] = "(name LIKE '%$search%' OR description LIKE '%$search%')";
}

// Durum filtresi
if ($statusFilter !== '') {
    $conditions[] = "status = " . intval($statusFilter);
}

// Koşulları SQL sorgusuna ekle
if (!empty($conditions)) {
    $sql .= " WHERE " . implode(" AND ", $conditions);
}

// Sıralama
$sql .= " ORDER BY sort_order ASC, name ASC";

// Sorguyu çalıştır
$result = $conn->query($sql);
$features = [];

// Sorgu başarısız olursa hata mesajı göster
if ($result === false) {
    $error = "SQL sorgusu başarısız oldu: " . $conn->error;
} else {
    while ($row = $result->fetch_assoc()) {
        $features[] = $row;
    }
}

// Aktif kullanıcı bilgilerini al
$user = getUser($_SESSION['user_id']);
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $pageTitle; ?> | <?php echo $settings['site_title'] ?? 'Dijital Menü'; ?></title>
    
    <!-- Favicon -->
    <?php if (!empty($settings['favicon'])): ?>
    <link rel="shortcut icon" href="../assets/img/<?php echo $settings['favicon']; ?>" type="image/x-icon">
    <?php endif; ?>
    
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    
    <!-- Font Awesome -->
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    
    <!-- Ana CSS -->
    <link href="../assets/css/style.css" rel="stylesheet">
    
    <!-- Özel Renkler -->
    <style>
        :root {
            --primary-color: <?php echo $settings['primary_color'] ?? '#3490dc'; ?>;
            --secondary-color: <?php echo $settings['secondary_color'] ?? '#38c172'; ?>;
            --text-color: <?php echo $settings['text_color'] ?? '#333333'; ?>;
            --background-color: <?php echo $settings['background_color'] ?? '#ffffff'; ?>;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <?php include 'includes/sidebar.php'; ?>
            
            <!-- Ana İçerik -->
            <div class="col-md-9 col-lg-10 admin-content">
                <!-- Üst Başlık -->
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h1 class="h3"><?php echo $pageTitle; ?></h1>
                    <?php if ($action === 'list'): ?>
                    <a href="features.php?action=add" class="btn btn-primary">
                        <i class="fas fa-plus-circle"></i> <?php echo __('admin.add_feature', [], $adminLanguage); ?>
                    </a>
                    <?php else: ?>
                    <a href="features.php" class="btn btn-secondary">
                        <i class="fas fa-arrow-left"></i> <?php echo __('admin.go_back', [], $adminLanguage); ?>
                    </a>
                    <?php endif; ?>
                </div>
                
                <?php if ($action === 'list'): ?>
                <!-- Arama ve Filtre -->
                <div class="card mb-4">
                    <div class="card-body">
                        <form method="get" action="features.php" class="row g-3">
                            <div class="col-md-6">
                                <div class="input-group">
                                    <input type="text" class="form-control" name="search" placeholder="<?php echo __('admin.search', [], $adminLanguage); ?>..." value="<?php echo isset($_GET['search']) ? htmlspecialchars($_GET['search']) : ''; ?>">
                                    <button class="btn btn-outline-secondary" type="submit">
                                        <i class="fas fa-search"></i>
                                    </button>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <select class="form-select" name="status">
                                    <option value=""><?php echo __('admin.all_status', [], $adminLanguage); ?></option>
                                    <option value="1" <?php echo (isset($_GET['status']) && $_GET['status'] == '1') ? 'selected' : ''; ?>><?php echo __('admin.active', [], $adminLanguage); ?></option>
                                    <option value="0" <?php echo (isset($_GET['status']) && $_GET['status'] == '0') ? 'selected' : ''; ?>><?php echo __('admin.inactive', [], $adminLanguage); ?></option>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <a href="features.php" class="btn btn-outline-secondary w-100">
                                    <i class="fas fa-redo"></i> <?php echo __('admin.reset', [], $adminLanguage); ?>
                                </a>
                            </div>
                        </form>
                    </div>
                </div>
                <?php endif; ?>
                
                <!-- Oturum Mesajı -->
                <?php if (isset($_SESSION['error_message'])): ?>
                <div class="alert alert-danger alert-dismissible fade show" role="alert">
                    <?php echo $_SESSION['error_message']; ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
                <?php unset($_SESSION['error_message']); ?>
                <?php endif; ?>
                
                <?php if (isset($_SESSION['success_message'])): ?>
                <div class="alert alert-success alert-dismissible fade show" role="alert">
                    <?php echo $_SESSION['success_message']; ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
                <?php unset($_SESSION['success_message']); ?>
                <?php endif; ?>
                
                <?php if (!empty($error)): ?>
                <div class="alert alert-danger"><?php echo $error; ?></div>
                <?php endif; ?>
                
                <?php if (!empty($success)): ?>
                <div class="alert alert-success"><?php echo $success; ?></div>
                <?php endif; ?>
                
                <!-- İçerik -->
                <div class="admin-card">
                    <?php if ($action === 'list'): ?>
                    <!-- Özellik Listesi -->
                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead>
                                <tr>
                                    <th class="d-none">ID</th>
                                    <th><?php echo __('admin.feature_icon', [], $adminLanguage); ?></th>
                                    <th><?php echo __('admin.name', [], $adminLanguage); ?></th>
                                    <th><?php echo __('admin.description', [], $adminLanguage); ?></th>
                                    <th><?php echo __('admin.feature_color', [], $adminLanguage); ?></th>
                                    <th><?php echo __('admin.sort_order', [], $adminLanguage); ?></th>
                                    <th><?php echo __('admin.status', [], $adminLanguage); ?></th>
                                    <th><?php echo __('admin.actions', [], $adminLanguage); ?></th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($features)): ?>
                                <tr>
                                    <td colspan="8" class="text-center"><?php echo __('admin.no_features', [], $adminLanguage); ?></td>
                                </tr>
                                <?php else: ?>
                                <?php foreach ($features as $feature): ?>
                                <tr>
                                    <td class="d-none"><?php echo isset($feature['id']) ? $feature['id'] : '0'; ?></td>
                                    <td>
                                        <?php if (isset($feature['icon']) && !empty($feature['icon'])): ?>
                                        <span class="badge" style="background-color: <?php echo isset($feature['color']) ? $feature['color'] : '#3490dc'; ?>; font-size: 1.2rem;">
                                            <i class="fas <?php echo strpos($feature['icon'], 'fa-') === 0 ? $feature['icon'] : 'fa-' . $feature['icon']; ?>"></i>
                                        </span>
                                        <?php else: ?>
                                        <span class="badge bg-secondary">
                                            <i class="fas fa-tag"></i>
                                        </span>
                                        <?php endif; ?>
                                    </td>
                                    <td><?php echo isset($feature['name']) ? $feature['name'] : 'N/A'; ?></td>
                                    <td><?php echo isset($feature['description']) ? $feature['description'] : ''; ?></td>
                                    <td>
                                        <div class="d-flex align-items-center">
                                            <div class="color-preview me-2" style="width: 20px; height: 20px; background-color: <?php echo isset($feature['color']) ? $feature['color'] : '#3490dc'; ?>; border-radius: 3px;"></div>
                                            <span><?php echo isset($feature['color']) ? $feature['color'] : '#3490dc'; ?></span>
                                        </div>
                                    </td>
                                    <td><?php echo isset($feature['sort_order']) ? $feature['sort_order'] : '0'; ?></td>
                                    <td>
                                        <?php if (isset($feature['status']) && $feature['status'] == 1): ?>
                                        <span class="badge bg-success"><?php echo __('admin.active', [], $adminLanguage); ?></span>
                                        <?php else: ?>
                                        <span class="badge bg-danger"><?php echo __('admin.inactive', [], $adminLanguage); ?></span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <a href="features.php?action=edit&id=<?php echo isset($feature['id']) ? $feature['id'] : '0'; ?>" class="btn btn-sm btn-primary">
                                            <i class="fas fa-edit"></i>
                                        </a>
                                        <a href="features.php?delete=1&id=<?php echo isset($feature['id']) ? $feature['id'] : '0'; ?>" class="btn btn-sm btn-danger" onclick="return confirm('<?php echo __('admin.confirm_delete', [], $adminLanguage); ?>');">
                                            <i class="fas fa-trash"></i>
                                        </a>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                    <?php else: ?>
                    <!-- Özellik Formu -->
                    <form method="post" action="<?php echo htmlspecialchars($_SERVER['PHP_SELF'] . '?action=' . $action . ($featureId ? '&id=' . $featureId : '')); ?>">
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="name" class="form-label"><?php echo __('admin.feature_name', [], $adminLanguage); ?> <span class="text-danger">*</span></label>
                                    <input type="text" class="form-control" id="name" name="name" value="<?php echo $feature['name'] ?? ''; ?>" required>
                                </div>
                                
                                <div class="mb-3">
                                    <label for="description" class="form-label"><?php echo __('admin.feature_description', [], $adminLanguage); ?></label>
                                    <textarea class="form-control" id="description" name="description" rows="3"><?php echo $feature['description'] ?? ''; ?></textarea>
                                </div>
                                
                                <div class="mb-3">
                                    <label for="sort_order" class="form-label"><?php echo __('admin.sort_order', [], $adminLanguage); ?></label>
                                    <input type="number" class="form-control" id="sort_order" name="sort_order" value="<?php echo $feature['sort_order'] ?? 0; ?>" min="0">
                                </div>
                                
                                <div class="mb-3 form-check">
                                    <input type="checkbox" class="form-check-input" id="status" name="status" <?php echo (isset($feature['status']) && $feature['status'] == 1) ? 'checked' : ''; ?>>
                                    <label class="form-check-label" for="status"><?php echo __('admin.active', [], $adminLanguage); ?></label>
                                </div>
                            </div>
                            
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="icon" class="form-label"><?php echo __('admin.feature_icon', [], $adminLanguage); ?></label>
                                    <div class="input-group">
                                        <span class="input-group-text">fa-</span>
                                        <input type="text" class="form-control" id="icon" name="icon" value="<?php echo $feature['icon'] ?? ''; ?>" placeholder="star, leaf, fire, etc.">
                                    </div>
                                    <small class="text-muted">Font Awesome ikon adını girin (fa- öneki olmadan). <a href="https://fontawesome.com/icons" target="_blank">İkonları görüntüle</a></small>
                                </div>
                                
                                <div class="mb-3">
                                    <label for="color" class="form-label"><?php echo __('admin.feature_color', [], $adminLanguage); ?></label>
                                    <div class="input-group">
                                        <input type="color" class="form-control form-control-color" id="color-picker" value="<?php echo $feature['color'] ?? '#3490dc'; ?>" title="Renk seçin">
                                        <input type="text" class="form-control" id="color" name="color" value="<?php echo $feature['color'] ?? '#3490dc'; ?>">
                                    </div>
                                </div>
                                
                                <div class="mb-3">
                                    <label class="form-label"><?php echo __('admin.preview', [], $adminLanguage); ?></label>
                                    <div class="p-3 border rounded">
                                        <div class="d-flex align-items-center">
                                            <span class="badge me-2" id="preview-badge" style="background-color: <?php echo $feature['color'] ?? '#3490dc'; ?>; font-size: 1.2rem;">
                                                <i class="fas <?php echo !empty($feature['icon']) ? (strpos($feature['icon'], 'fa-') === 0 ? $feature['icon'] : 'fa-' . $feature['icon']) : 'fa-tag'; ?>" id="preview-icon"></i>
                                            </span>
                                            <span id="preview-name"><?php echo $feature['name'] ?? 'Özellik Adı'; ?></span>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="mt-4">
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-save"></i> <?php echo __('admin.save', [], $adminLanguage); ?>
                            </button>
                            <a href="features.php" class="btn btn-secondary">
                                <i class="fas fa-times"></i> <?php echo __('admin.cancel', [], $adminLanguage); ?>
                            </a>
                        </div>
                    </form>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Bootstrap JS -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    
    <!-- jQuery -->
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    
    <!-- Ana JS -->
    <script src="../assets/js/script.js"></script>
    
    <?php if ($action !== 'list'): ?>
    <!-- Özellik Önizleme JS -->
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const nameInput = document.getElementById('name');
            const iconInput = document.getElementById('icon');
            const colorInput = document.getElementById('color');
            const colorPicker = document.getElementById('color-picker');
            const previewName = document.getElementById('preview-name');
            const previewIcon = document.getElementById('preview-icon');
            const previewBadge = document.getElementById('preview-badge');
            
            // İsim değiştiğinde önizlemeyi güncelle
            nameInput.addEventListener('input', function() {
                previewName.textContent = this.value || 'Özellik Adı';
            });
            
            // İkon değiştiğinde önizlemeyi güncelle
            iconInput.addEventListener('input', function() {
                previewIcon.className = 'fas ' + (this.value ? 'fa-' + this.value : 'fa-tag');
            });
            
            // Renk değiştiğinde önizlemeyi güncelle
            colorInput.addEventListener('input', function() {
                previewBadge.style.backgroundColor = this.value;
                colorPicker.value = this.value;
            });
            
            // Renk seçici değiştiğinde önizlemeyi güncelle
            colorPicker.addEventListener('input', function() {
                previewBadge.style.backgroundColor = this.value;
                colorInput.value = this.value;
            });
        });
    </script>
    <?php endif; ?>
</body>
</html>
