/**
 * Puanlama Sistemi JavaScript
 */

document.addEventListener('DOMContentLoaded', function() {
    // Puanlama modallarını başlat
    initRatingModals();
    
    // Puanlama formlarını başlat
    initRatingForms();
});

/**
 * Puanlama modallarını başlat
 */
function initRatingModals() {
    // Puanlama butonlarına tıklama olayı ekle
    const ratingButtons = document.querySelectorAll('.btn-show-ratings');
    ratingButtons.forEach(button => {
        button.addEventListener('click', function() {
            const itemId = this.dataset.itemId;
            const modal = document.getElementById('ratingModal-' + itemId);
            if (modal) {
                modal.style.display = 'block';
                document.body.style.overflow = 'hidden'; // Sayfanın kaydırılmasını engelle
            }
        });
    });
    
    // Kapatma butonlarına tıklama olayı ekle
    const closeButtons = document.querySelectorAll('.rating-modal-close');
    closeButtons.forEach(button => {
        button.addEventListener('click', function() {
            const modal = this.closest('.rating-modal');
            if (modal) {
                modal.style.display = 'none';
                document.body.style.overflow = ''; // Sayfanın kaydırılmasını tekrar etkinleştir
            }
        });
    });
    
    // Modal dışına tıklama olayı ekle
    const modals = document.querySelectorAll('.rating-modal');
    modals.forEach(modal => {
        modal.addEventListener('click', function(e) {
            if (e.target === this) {
                this.style.display = 'none';
                document.body.style.overflow = ''; // Sayfanın kaydırılmasını tekrar etkinleştir
            }
        });
    });
}

/**
 * Puanlama formlarını başlat
 */
function initRatingForms() {
    // Puanlama formlarına gönderme olayı ekle
    const ratingForms = document.querySelectorAll('.rating-form form');
    ratingForms.forEach(form => {
        form.addEventListener('submit', function(e) {
            e.preventDefault();
            
            // Form verilerini al
            const formData = new FormData(this);
            const itemId = formData.get('item_id');
            const rating = formData.get('rating');
            const userName = formData.get('user_name');
            const comment = formData.get('comment');
            
            // Validasyon
            if (!rating) {
                showRatingError(form.parentElement, 'Lütfen bir puan seçin.');
                return;
            }
            
            // Doğrudan fetch API kullanarak gönder
            fetch('rate_item.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: new URLSearchParams({
                    item_id: itemId,
                    rating: rating,
                    user_name: 'Misafir', // Kullanıcı adı alanı kaldırıldı, her zaman "Misafir" olarak gönder
                    comment: comment || ''
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Başarılı mesajı göster
                    showRatingSuccess(form.parentElement, data.message || 'Puanlamanız için teşekkürler!');
                    
                    // Formu sıfırla
                    form.reset();
                    
                    // Yıldız seçimini sıfırla
                    const stars = form.querySelectorAll('.rating-selector input');
                    stars.forEach(star => {
                        star.checked = false;
                    });
                    
                    // Sayfa yenileme olmadan puanlamayı güncelle
                    if (data.average) {
                        updateItemRating(itemId, data.average, data.count);
                    }
                    
                    // Puanlama listesini güncelle
                    if (data.html) {
                        updateRatingsList(itemId, data.html);
                    }
                } else {
                    // Hata mesajı göster
                    showRatingError(form.parentElement, data.message || 'Bir hata oluştu. Lütfen tekrar deneyin.');
                    
                    // Eğer kullanıcı daha önce bu ürüne puan vermişse, formu devre dışı bırak
                    if (data.message === 'Bu ürüne daha önce puan verdiniz.') {
                        // Form elemanlarını devre dışı bırak
                        const stars = form.querySelectorAll('.rating-selector input');
                        stars.forEach(star => {
                            star.disabled = true;
                        });
                        
                        const commentTextarea = form.querySelector('textarea[name="comment"]');
                        if (commentTextarea) {
                            commentTextarea.disabled = true;
                        }
                        
                        const submitButton = form.querySelector('button[type="submit"]');
                        if (submitButton) {
                            submitButton.disabled = true;
                        }
                    }
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showRatingError(form.parentElement, 'Bağlantı hatası. Lütfen tekrar deneyin.');
            });
        });
    });
}

/**
 * Puanlama hata mesajı göster
 */
function showRatingError(form, message) {
    let errorDiv = form.querySelector('.rating-error');
    
    if (!errorDiv) {
        errorDiv = document.createElement('div');
        errorDiv.className = 'alert alert-danger rating-error';
        form.insertBefore(errorDiv, form.firstChild);
    }
    
    errorDiv.textContent = message;
    errorDiv.style.display = 'block';
    
    // 5 saniye sonra hata mesajını gizle
    setTimeout(() => {
        errorDiv.style.display = 'none';
    }, 5000);
}

/**
 * Puanlama başarı mesajı göster
 */
function showRatingSuccess(form, message) {
    let successDiv = form.querySelector('.rating-success');
    
    if (!successDiv) {
        successDiv = document.createElement('div');
        successDiv.className = 'alert alert-success rating-success';
        form.insertBefore(successDiv, form.firstChild);
    }
    
    successDiv.textContent = message;
    successDiv.style.display = 'block';
    
    // 5 saniye sonra başarı mesajını gizle
    setTimeout(() => {
        successDiv.style.display = 'none';
    }, 5000);
}

/**
 * Ürün puanlamasını güncelle
 */
function updateItemRating(itemId, average, count) {
    // Ürün kartındaki puanlama özetini güncelle
    const ratingContainer = document.querySelector(`.menu-item-rating[data-item-id="${itemId}"]`);
    if (ratingContainer) {
        // Yıldızları güncelle
        const starsContainer = ratingContainer.querySelector('.star-rating');
        if (starsContainer) {
            starsContainer.innerHTML = generateStarRating(average);
        }
        
        // Ortalama puanı güncelle
        const averageElement = ratingContainer.querySelector('.rating-average');
        if (averageElement) {
            averageElement.textContent = average;
        }
        
        // Puanlama sayısını güncelle
        const countElement = ratingContainer.querySelector('.rating-count');
        if (countElement) {
            countElement.textContent = `(${count})`;
        }
    }
    
    // Modal içindeki puanlama özetini güncelle
    const modal = document.getElementById(`ratingModal-${itemId}`);
    if (modal) {
        const modalStarsContainer = modal.querySelector('.star-rating');
        if (modalStarsContainer) {
            modalStarsContainer.innerHTML = generateStarRating(average);
        }
        
        const modalAverageElement = modal.querySelector('.rating-average');
        if (modalAverageElement) {
            modalAverageElement.textContent = average;
        }
        
        const modalCountElement = modal.querySelector('.rating-count');
        if (modalCountElement) {
            modalCountElement.textContent = `(${count})`;
        }
    }
}

/**
 * Puanlama listesini güncelle
 */
function updateRatingsList(itemId, html) {
    const ratingsList = document.querySelector(`#ratingModal-${itemId} .ratings-list`);
    if (ratingsList) {
        ratingsList.innerHTML = html;
    }
}

/**
 * Yıldız HTML'i oluştur
 */
function generateStarRating(rating) {
    const fullStars = Math.floor(rating);
    const halfStar = rating - fullStars;
    let html = '';
    
    // Tam yıldızlar
    for (let i = 0; i < fullStars; i++) {
        html += '<i class="fas fa-star"></i>';
    }
    
    // Yarım yıldız
    if (halfStar >= 0.25 && halfStar < 0.75) {
        html += '<i class="fas fa-star-half-alt"></i>';
    } else if (halfStar >= 0.75) {
        html += '<i class="fas fa-star"></i>';
    }
    
    // Boş yıldızlar
    const emptyStars = 5 - Math.ceil(rating);
    for (let i = 0; i < emptyStars; i++) {
        html += '<i class="far fa-star"></i>';
    }
    
    return html;
}
