<?php
/**
 * Dijital Menü - Dil Fonksiyonları
 */

/**
 * Dil dosyasını yükle
 * 
 * @param string $code Dil kodu
 * @return array Dil verileri
 */
function loadLanguage($code = null) {
    // Dil kodu belirtilmemişse, oturumdan veya varsayılan dili al
    if ($code === null) {
        $code = isset($_SESSION['language']) ? $_SESSION['language'] : 'tr';
    }
    
    // Dil dosyası yolu
    $langFile = __DIR__ . '/../languages/' . $code . '.json';
    
    // Dil dosyası yoksa, varsayılan dili yükle
    if (!file_exists($langFile)) {
        $langFile = __DIR__ . '/../languages/tr.json';
        $code = 'tr';
    }
    
    // Dil dosyasını yükle
    $translations = json_decode(file_get_contents($langFile), true);
    
    return $translations;
}

/**
 * Çeviri al
 * 
 * @param string $key Çeviri anahtarı
 * @param array $params Değiştirilecek parametreler
 * @param string $code Dil kodu
 * @return string Çeviri
 */
function __($key, $params = [], $code = null) {
    static $translations = null;
    
    // Çevirileri yükle (ilk çağrıda)
    if ($translations === null) {
        $translations = loadLanguage($code);
    }
    
    // Nokta notasyonu ile alt dizilere erişim (örn: "site.menu")
    $parts = explode('.', $key);
    $value = $translations;
    
    foreach ($parts as $part) {
        if (isset($value[$part])) {
            $value = $value[$part];
        } else {
            return $key; // Çeviri bulunamadı, anahtarı döndür
        }
    }
    
    // Parametreleri değiştir
    if (!empty($params) && is_string($value)) {
        foreach ($params as $param => $replacement) {
            $value = str_replace(':' . $param, $replacement, $value);
        }
    }
    
    return $value;
}

/**
 * Dil listesini al
 * 
 * @return array Dil listesi
 */
function getAvailableLanguages() {
    $languages = [];
    $langDir = __DIR__ . '/../languages/';
    
    if (is_dir($langDir)) {
        $files = scandir($langDir);
        
        foreach ($files as $file) {
            if (pathinfo($file, PATHINFO_EXTENSION) === 'json') {
                $code = pathinfo($file, PATHINFO_FILENAME);
                $langData = json_decode(file_get_contents($langDir . $file), true);
                
                $languages[$code] = [
                    'code' => $code,
                    'name' => $langData['meta']['name'] ?? ucfirst($code),
                    'is_default' => $langData['meta']['is_default'] ?? false,
                    'status' => $langData['meta']['status'] ?? true
                ];
            }
        }
    }
    
    return $languages;
}

/**
 * Dil değiştir
 * 
 * @param string $code Dil kodu
 * @return void
 */
function changeLanguage($code) {
    // Dil dosyası var mı kontrol et
    $langFile = __DIR__ . '/../languages/' . $code . '.json';
    
    if (file_exists($langFile)) {
        $_SESSION['language'] = $code;
    }
}

/**
 * Admin paneli dilini değiştir
 * 
 * @param string $code Dil kodu
 * @return void
 */
function changeAdminLanguage($code) {
    // Dil dosyası var mı kontrol et
    $langFile = __DIR__ . '/../languages/' . $code . '.json';
    
    if (file_exists($langFile)) {
        $_SESSION['admin_language'] = $code;
    }
}

/**
 * Admin paneli dilini al
 * 
 * @return string Dil kodu
 */
function getAdminLanguage() {
    // Admin dili belirtilmemişse, oturumdan veya varsayılan dili al
    if (isset($_SESSION['admin_language'])) {
        return $_SESSION['admin_language'];
    }
    
    // Varsayılan dili döndür
    return getDefaultLanguage();
}

/**
 * Dil dosyasını güncelle
 * 
 * @param string $code Dil kodu
 * @param array $translations Çeviriler
 * @param string $section Bölüm (null ise tüm dosya)
 * @return bool İşlem sonucu
 */
function updateLanguageFile($code, $translations, $section = null) {
    $langFile = __DIR__ . '/../languages/' . $code . '.json';
    
    // Dil dosyası yoksa oluştur
    if (!file_exists($langFile)) {
        return createLanguageFile($code, $translations);
    }
    
    // Mevcut çevirileri al
    $currentTranslations = json_decode(file_get_contents($langFile), true);
    
    // Belirli bir bölüm güncelleniyor
    if ($section !== null) {
        if (!isset($currentTranslations[$section])) {
            $currentTranslations[$section] = [];
        }
        
        // Bölüm çevirilerini güncelle
        foreach ($translations as $key => $value) {
            $currentTranslations[$section][$key] = $value;
        }
    } else {
        // Tüm çevirileri güncelle
        $currentTranslations = array_merge($currentTranslations, $translations);
    }
    
    // Dosyayı güncelle
    return file_put_contents($langFile, json_encode($currentTranslations, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE)) !== false;
}

/**
 * Dil dosyasını oluştur
 * 
 * @param string $code Dil kodu
 * @param array $translations Çeviriler
 * @return bool İşlem sonucu
 */
function createLanguageFile($code, $translations) {
    $langFile = __DIR__ . '/../languages/' . $code . '.json';
    
    // Meta bilgilerini ekle
    if (!isset($translations['meta'])) {
        $translations['meta'] = [
            'name' => ucfirst($code),
            'code' => $code,
            'is_default' => ($code === 'tr'),
            'status' => true
        ];
    }
    
    // Dosyayı oluştur
    return file_put_contents($langFile, json_encode($translations, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE)) !== false;
}

/**
 * Varsayılan dil kodunu al
 * 
 * @return string Varsayılan dil kodu
 */
function getDefaultLanguage() {
    $languages = getAvailableLanguages();
    
    foreach ($languages as $code => $language) {
        if ($language['is_default']) {
            return $code;
        }
    }
    
    return 'tr'; // Varsayılan olarak Türkçe
}

/**
 * Dil dosyasını varsayılan yap
 * 
 * @param string $code Dil kodu
 * @return bool İşlem sonucu
 */
function setDefaultLanguage($code) {
    $languages = getAvailableLanguages();
    $success = true;
    
    // Tüm dilleri güncelle
    foreach ($languages as $langCode => $language) {
        $langFile = __DIR__ . '/../languages/' . $langCode . '.json';
        $langData = json_decode(file_get_contents($langFile), true);
        
        // Varsayılan durumunu güncelle
        $langData['meta']['is_default'] = ($langCode === $code);
        
        // Dosyayı güncelle
        if (file_put_contents($langFile, json_encode($langData, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE)) === false) {
            $success = false;
        }
    }
    
    return $success;
}
