<?php
/**
 * Puanlama Sistemi Fonksiyonları
 */

/**
 * Kullanıcının daha önce bu ürüne puan verip vermediğini kontrol et
 * 
 * @param int $itemId Ürün ID
 * @param string $ipAddress IP adresi
 * @return bool Daha önce puan verilmiş mi
 */
function hasUserRatedItem($itemId, $ipAddress) {
    global $conn;
    
    $stmt = $conn->prepare("SELECT COUNT(*) as count FROM ratings WHERE item_id = ? AND ip_address = ?");
    $stmt->bind_param("is", $itemId, $ipAddress);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        $row = $result->fetch_assoc();
        return $row['count'] > 0;
    }
    
    return false;
}

/**
 * Puanlama ekle
 * 
 * @param int $itemId Ürün ID
 * @param int $rating Puan (1-5)
 * @param string $comment Yorum
 * @param string $userName Kullanıcı adı
 * @param string $ipAddress IP adresi
 * @return bool İşlem sonucu
 */
function addRating($itemId, $rating, $comment, $userName, $ipAddress = null) {
    global $conn;
    
    // IP adresi yoksa, kullanıcının IP adresini al
    if ($ipAddress === null) {
        $ipAddress = $_SERVER['REMOTE_ADDR'];
    }
    
    // Kullanıcının daha önce bu ürüne puan verip vermediğini kontrol et
    if (hasUserRatedItem($itemId, $ipAddress)) {
        return false;
    }
    
    // Ayarlardan onay gerekip gerekmediğini kontrol et
    $requireApproval = getSetting('require_approval') ?? '1';
    $status = ($requireApproval == '1') ? 'pending' : 'approved';
    
    // Puanlamayı ekle
    $stmt = $conn->prepare("INSERT INTO ratings (item_id, rating, comment, user_name, status, ip_address) VALUES (?, ?, ?, ?, ?, ?)");
    $stmt->bind_param("iissss", $itemId, $rating, $comment, $userName, $status, $ipAddress);
    
    return $stmt->execute();
}

/**
 * Ürün puanlamalarını al
 * 
 * @param int $itemId Ürün ID
 * @param string $status Durum filtresi (null ise tüm puanlamalar)
 * @return array Puanlamalar
 */
function getItemRatings($itemId, $status = 'approved') {
    global $conn;
    
    $sql = "SELECT * FROM ratings WHERE item_id = ?";
    
    if ($status !== null) {
        $sql .= " AND status = ?";
    }
    
    $sql .= " ORDER BY created_at DESC";
    
    $stmt = $conn->prepare($sql);
    
    if ($status !== null) {
        $stmt->bind_param("is", $itemId, $status);
    } else {
        $stmt->bind_param("i", $itemId);
    }
    
    $stmt->execute();
    $result = $stmt->get_result();
    $ratings = [];
    
    while ($row = $result->fetch_assoc()) {
        $ratings[] = $row;
    }
    
    return $ratings;
}

/**
 * Tüm puanlamaları al
 * 
 * @param string $status Durum filtresi (null ise tüm puanlamalar)
 * @param string $search Arama filtresi (null ise tüm puanlamalar)
 * @param int $rating Puan filtresi (null ise tüm puanlar)
 * @param int $categoryId Kategori ID filtresi (null ise tüm kategoriler)
 * @param string $startDate Başlangıç tarihi (null ise tüm tarihler)
 * @param string $endDate Bitiş tarihi (null ise tüm tarihler)
 * @return array Puanlamalar
 */
function getAllRatings($status = null, $search = null, $rating = null, $categoryId = null, $startDate = null, $endDate = null) {
    global $conn;
    
    $sql = "SELECT r.*, i.name as item_name, i.category_id, c.name as category_name 
            FROM ratings r 
            JOIN items i ON r.item_id = i.id 
            JOIN categories c ON i.category_id = c.id";
    
    $conditions = [];
    $params = [];
    $types = "";
    
    if ($status !== null) {
        $conditions[] = "r.status = ?";
        $params[] = $status;
        $types .= "s";
    }
    
    if ($search !== null) {
        $searchTerm = "%$search%";
        $conditions[] = "(i.name LIKE ? OR c.name LIKE ? OR r.user_name LIKE ?)";
        $params[] = $searchTerm;
        $params[] = $searchTerm;
        $params[] = $searchTerm;
        $types .= "sss";
    }
    
    if ($rating !== null && $rating >= 1 && $rating <= 5) {
        $conditions[] = "r.rating = ?";
        $params[] = $rating;
        $types .= "i";
    }
    
    if ($categoryId !== null && $categoryId > 0) {
        $conditions[] = "i.category_id = ?";
        $params[] = $categoryId;
        $types .= "i";
    }
    
    // Tarih filtreleri
    if ($startDate !== null) {
        $conditions[] = "DATE(r.created_at) >= ?";
        $params[] = $startDate;
        $types .= "s";
    }
    
    if ($endDate !== null) {
        $conditions[] = "DATE(r.created_at) <= ?";
        $params[] = $endDate;
        $types .= "s";
    }
    
    if (!empty($conditions)) {
        $sql .= " WHERE " . implode(" AND ", $conditions);
    }
    
    $sql .= " ORDER BY r.created_at DESC";
    
    $stmt = $conn->prepare($sql);
    
    if (!empty($params)) {
        $stmt->bind_param($types, ...$params);
    }
    
    $stmt->execute();
    $result = $stmt->get_result();
    $ratings = [];
    
    while ($row = $result->fetch_assoc()) {
        $ratings[] = $row;
    }
    
    return $ratings;
}

/**
 * Puanlama durumunu güncelle
 * 
 * @param int $ratingId Puanlama ID
 * @param string $status Yeni durum (approved/rejected)
 * @param int $reviewedBy İnceleyenin kullanıcı ID'si
 * @return bool İşlem sonucu
 */
function updateRatingStatus($ratingId, $status, $reviewedBy) {
    global $conn;
    
    $stmt = $conn->prepare("UPDATE ratings SET status = ?, reviewed_by = ?, reviewed_at = NOW() WHERE id = ?");
    $stmt->bind_param("sii", $status, $reviewedBy, $ratingId);
    
    return $stmt->execute();
}

/**
 * Puanlamayı sil
 * 
 * @param int $ratingId Puanlama ID
 * @return bool İşlem sonucu
 */
function deleteRating($ratingId) {
    global $conn;
    
    $stmt = $conn->prepare("DELETE FROM ratings WHERE id = ?");
    $stmt->bind_param("i", $ratingId);
    
    return $stmt->execute();
}

/**
 * Ürünün ortalama puanını al
 * 
 * @param int $itemId Ürün ID
 * @return float|null Ortalama puan
 */
function getItemAverageRating($itemId) {
    global $conn;
    
    $stmt = $conn->prepare("SELECT AVG(rating) as average FROM ratings WHERE item_id = ? AND status = 'approved'");
    $stmt->bind_param("i", $itemId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        $row = $result->fetch_assoc();
        return round($row['average'], 1);
    }
    
    return null;
}

/**
 * Ürünün puanlama sayısını al
 * 
 * @param int $itemId Ürün ID
 * @return int Puanlama sayısı
 */
function getItemRatingCount($itemId) {
    global $conn;
    
    $stmt = $conn->prepare("SELECT COUNT(*) as count FROM ratings WHERE item_id = ? AND status = 'approved'");
    $stmt->bind_param("i", $itemId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        $row = $result->fetch_assoc();
        return $row['count'];
    }
    
    return 0;
}

/**
 * Yıldız HTML'i oluştur
 * 
 * @param float $rating Puan
 * @param int $maxRating Maksimum puan
 * @return string Yıldız HTML'i
 */
function generateStarRating($rating, $maxRating = 5) {
    $html = '<div class="star-rating">';
    
    // Tam yıldızlar
    $fullStars = floor($rating);
    for ($i = 0; $i < $fullStars; $i++) {
        $html .= '<i class="fas fa-star"></i>';
    }
    
    // Yarım yıldız
    $halfStar = $rating - $fullStars;
    if ($halfStar >= 0.25 && $halfStar < 0.75) {
        $html .= '<i class="fas fa-star-half-alt"></i>';
        $i++;
    } elseif ($halfStar >= 0.75) {
        $html .= '<i class="fas fa-star"></i>';
        $i++;
    }
    
    // Boş yıldızlar
    for ($j = $i; $j < $maxRating; $j++) {
        $html .= '<i class="far fa-star"></i>';
    }
    
    $html .= '</div>';
    
    return $html;
}
