<?php
session_start();
require_once '../config/db.php';
require_once '../includes/functions.php';

// Admin giriş kontrolü
if (!isLoggedIn()) {
    redirectToLogin();
}

// Site ayarlarını getir
$siteSettings = getSiteSettings($conn);
$siteTitle = $siteSettings['site_title'] ?? 'Restoran Menü';
$faviconImage = $siteSettings['favicon_image'] ?? '';

// Kategori silme işlemi
if (isset($_GET['delete']) && !empty($_GET['delete'])) {
    $id = intval($_GET['delete']);
    
    // Önce bu kategoriye ait ürünleri kontrol et
    $stmt = $conn->prepare("SELECT COUNT(*) as count FROM products WHERE category_id = :id");
    $stmt->bindParam(':id', $id, PDO::PARAM_INT);
    $stmt->execute();
    $result = $stmt->fetch();
    
    if ($result['count'] > 0) {
        $delete_error = "Bu kategoriye ait ürünler bulunduğu için silinemez. Önce ürünleri silmelisiniz.";
    } else {
        // Kategori resmini kontrol et ve sil
        $stmt = $conn->prepare("SELECT image FROM categories WHERE id = :id");
        $stmt->bindParam(':id', $id, PDO::PARAM_INT);
        $stmt->execute();
        $category = $stmt->fetch();
        
        // Eğer özel yüklenmiş bir resim varsa sil
        if (!empty($category['image']) && !filter_var($category['image'], FILTER_VALIDATE_URL)) {
            $image_path = "../uploads/" . $category['image'];
            if (file_exists($image_path)) {
                unlink($image_path);
            }
        }
        
        // Kategoriyi sil
        $stmt = $conn->prepare("DELETE FROM categories WHERE id = :id");
        $stmt->bindParam(':id', $id, PDO::PARAM_INT);
        $stmt->execute();
        
        $delete_success = "Kategori başarıyla silindi.";
    }
}

// Sıralama işlemi
if (isset($_GET['move']) && !empty($_GET['move']) && isset($_GET['direction'])) {
    $id = intval($_GET['move']);
    $direction = $_GET['direction'];
    
    // Kategori bilgilerini al
    $stmt = $conn->prepare("SELECT id, sort_order FROM categories WHERE id = :id");
    $stmt->bindParam(':id', $id, PDO::PARAM_INT);
    $stmt->execute();
    $category = $stmt->fetch();
    
    if ($category) {
        $current_order = $category['sort_order'];
        $new_order = ($direction === 'up') ? $current_order - 1 : $current_order + 1;
        
        // Yeni sıra numarası 0'dan küçük olamaz
        if ($new_order >= 0) {
            // Diğer kategoriyi güncelle (yer değiştirme)
            $stmt = $conn->prepare("UPDATE categories SET sort_order = :current_order WHERE sort_order = :new_order");
            $stmt->bindParam(':current_order', $current_order, PDO::PARAM_INT);
            $stmt->bindParam(':new_order', $new_order, PDO::PARAM_INT);
            $stmt->execute();
            
            // Bu kategoriyi güncelle
            $stmt = $conn->prepare("UPDATE categories SET sort_order = :new_order WHERE id = :id");
            $stmt->bindParam(':new_order', $new_order, PDO::PARAM_INT);
            $stmt->bindParam(':id', $id, PDO::PARAM_INT);
            $stmt->execute();
            
            $sort_success = "Kategori sıralaması güncellendi.";
        } else {
            $sort_error = "Kategori zaten en üstte.";
        }
    }
}

// Kategorileri getir
$stmt = $conn->prepare("SELECT * FROM categories ORDER BY sort_order ASC");
$stmt->execute();
$categories = $stmt->fetchAll();

// Kategori için varsayılan resimler
$categoryImages = [
    'Başlangıçlar' => 'https://images.unsplash.com/photo-1546241072-48010ad2862c?q=80&w=500',
    'Ana Yemekler' => 'https://images.unsplash.com/photo-1544025162-d76694265947?q=80&w=500',
    'Tatlılar' => 'https://images.unsplash.com/photo-1563805042-7684c019e1cb?q=80&w=500',
    'İçecekler' => 'https://images.unsplash.com/photo-1544145945-f90425340c7e?q=80&w=500'
];
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Kategoriler - <?= $siteTitle; ?></title>
    <?php if (!empty($faviconImage)): ?>
        <?php if (filter_var($faviconImage, FILTER_VALIDATE_URL)): ?>
            <link rel="icon" href="<?= $faviconImage; ?>" type="image/x-icon">
            <link rel="shortcut icon" href="<?= $faviconImage; ?>" type="image/x-icon">
        <?php else: ?>
            <link rel="icon" href="../uploads/<?= $faviconImage; ?>?v=<?= time(); ?>" type="image/x-icon">
            <link rel="shortcut icon" href="../uploads/<?= $faviconImage; ?>?v=<?= time(); ?>" type="image/x-icon">
        <?php endif; ?>
    <?php else: ?>
        <link rel="icon" href="../assets/img/favicon.ico" type="image/x-icon">
        <link rel="shortcut icon" href="../assets/img/favicon.ico" type="image/x-icon">
    <?php endif; ?>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <link rel="stylesheet" href="../assets/css/admin.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/sortablejs@1.15.0/Sortable.min.css">
</head>
<body>
    <?php include 'includes/header.php'; ?>

    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4 py-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">Kategoriler</h1>
                    <div class="btn-toolbar mb-2 mb-md-0">
                        <a href="add_category.php" class="btn btn-sm btn-primary">
                            <i class="fas fa-plus-circle me-1"></i> Yeni Kategori Ekle
                        </a>
                    </div>
                </div>
                
                <?php if (isset($delete_error)): ?>
                    <div class="alert alert-danger"><?= $delete_error; ?></div>
                <?php endif; ?>
                
                <?php if (isset($delete_success)): ?>
                    <div class="alert alert-success"><?= $delete_success; ?></div>
                <?php endif; ?>
                
                <?php if (isset($sort_error)): ?>
                    <div class="alert alert-danger"><?= $sort_error; ?></div>
                <?php endif; ?>
                
                <?php if (isset($sort_success)): ?>
                    <div class="alert alert-success"><?= $sort_success; ?></div>
                <?php endif; ?>
                
                <div class="card">
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-striped table-hover">
                                <thead>
                                    <tr>
                                        <th width="5%">Sıra</th>
                                        <th width="5%">ID</th>
                                        <th width="10%">Resim</th>
                                        <th width="20%">Kategori Adı</th>
                                        <th width="30%">Açıklama</th>
                                        <th width="10%">Durum</th>
                                        <th width="20%">İşlemler</th>
                                    </tr>
                                </thead>
                                <tbody id="sortable-categories">
                                    <?php if (count($categories) > 0): ?>
                                        <?php foreach ($categories as $category): ?>
                                            <tr class="sortable-item" data-id="<?= $category['id']; ?>">
                                                <td>
                                                    <div class="d-flex">
                                                        <span class="sortable-handle me-2"><i class="fas fa-grip-vertical"></i></span>
                                                        <div class="btn-group-vertical btn-group-sm">
                                                            <a href="categories.php?move=<?= $category['id']; ?>&direction=up" class="btn btn-outline-secondary btn-sm sort-btn">
                                                                <i class="fas fa-chevron-up"></i>
                                                            </a>
                                                            <a href="categories.php?move=<?= $category['id']; ?>&direction=down" class="btn btn-outline-secondary btn-sm sort-btn">
                                                                <i class="fas fa-chevron-down"></i>
                                                            </a>
                                                        </div>
                                                    </div>
                                                </td>
                                                <td><?= $category['id']; ?></td>
                                                <td>
                                                    <?php 
                                                    if (!empty($category['image'])) {
                                                        if (filter_var($category['image'], FILTER_VALIDATE_URL)) {
                                                            // Unsplash veya harici URL
                                                            echo '<img src="' . $category['image'] . '" alt="' . $category['name'] . '" width="50" height="50" class="img-thumbnail">';
                                                        } else {
                                                            // Yüklenmiş resim
                                                            echo '<img src="../uploads/' . $category['image'] . '?v=' . time() . '" alt="' . $category['name'] . '" width="50" height="50" class="img-thumbnail">';
                                                        }
                                                    } else {
                                                        // Varsayılan resim
                                                        $defaultImg = isset($categoryImages[$category['name']]) ? 
                                                            $categoryImages[$category['name']] : 
                                                            'https://images.unsplash.com/photo-1555939594-58d7cb561ad1?q=80&w=500';
                                                        echo '<img src="' . $defaultImg . '" alt="' . $category['name'] . '" width="50" height="50" class="img-thumbnail">';
                                                    }
                                                    ?>
                                                </td>
                                                <td><?= $category['name']; ?></td>
                                                <td><?= substr($category['description'], 0, 50) . (strlen($category['description']) > 50 ? '...' : ''); ?></td>
                                                <td>
                                                    <?php if ($category['status'] == 1): ?>
                                                        <span class="badge bg-success">Aktif</span>
                                                    <?php else: ?>
                                                        <span class="badge bg-danger">Pasif</span>
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <a href="edit_category.php?id=<?= $category['id']; ?>" class="btn btn-sm btn-primary">
                                                        <i class="fas fa-edit"></i>
                                                    </a>
                                                    <a href="categories.php?delete=<?= $category['id']; ?>" class="btn btn-sm btn-danger" onclick="return confirm('Bu kategoriyi silmek istediğinize emin misiniz?')">
                                                        <i class="fas fa-trash"></i>
                                                    </a>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    <?php else: ?>
                                        <tr>
                                            <td colspan="7" class="text-center">Henüz kategori bulunmamaktadır.</td>
                                        </tr>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/sortablejs@1.15.0/Sortable.min.js"></script>
    <script src="../assets/js/admin.js"></script>
    <script>
        // Sürükle bırak sıralama
        document.addEventListener('DOMContentLoaded', function() {
            const sortableList = document.getElementById('sortable-categories');
            if (sortableList) {
                new Sortable(sortableList, {
                    handle: '.sortable-handle',
                    animation: 150,
                    onEnd: function(evt) {
                        const items = Array.from(sortableList.querySelectorAll('.sortable-item'));
                        const ids = items.map(item => item.dataset.id);
                        
                        // AJAX ile sıralamayı güncelle
                        fetch('update_category_order.php', {
                            method: 'POST',
                            headers: {
                                'Content-Type': 'application/json',
                            },
                            body: JSON.stringify({ ids: ids })
                        })
                        .then(response => response.json())
                        .then(data => {
                            if (data.success) {
                                // Başarılı mesajı göster
                                const alert = document.createElement('div');
                                alert.className = 'alert alert-success';
                                alert.textContent = 'Kategori sıralaması güncellendi.';
                                
                                const mainContent = document.querySelector('main');
                                mainContent.insertBefore(alert, mainContent.firstChild.nextSibling);
                                
                                // 3 saniye sonra mesajı kaldır
                                setTimeout(() => {
                                    alert.remove();
                                }, 3000);
                            }
                        })
                        .catch(error => {
                            console.error('Error:', error);
                        });
                    }
                });
            }
        });
    </script>
</body>
</html>