<?php
session_start();
require_once '../config/db.php';
require_once '../includes/functions.php';

// Cache control headers to prevent browser caching
header("Cache-Control: no-store, no-cache, must-revalidate, max-age=0");
header("Cache-Control: post-check=0, pre-check=0", false);
header("Pragma: no-cache");

// Admin giriş kontrolü
if (!isLoggedIn()) {
    redirectToLogin();
}

// Site ayarlarını getir
$siteSettings = getSiteSettings($conn);
$siteTitle = $siteSettings['site_title'] ?? 'Restoran Menü';
$faviconImage = $siteSettings['favicon_image'] ?? '';

// Kategori ID'sini al
$id = isset($_GET['id']) ? intval($_GET['id']) : 0;

// Kategori bilgilerini getir
$stmt = $conn->prepare("SELECT * FROM categories WHERE id = :id");
$stmt->bindParam(':id', $id, PDO::PARAM_INT);
$stmt->execute();
$category = $stmt->fetch();

// Kategori bulunamadıysa kategoriler sayfasına yönlendir
if (!$category) {
    header("Location: categories.php");
    exit;
}

$errors = [];
$success = false;

// Form gönderildiğinde
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Form verilerini al
    $name = isset($_POST['name']) ? sanitize($_POST['name']) : '';
    $description = isset($_POST['description']) ? sanitize($_POST['description']) : '';
    $sort_order = isset($_POST['sort_order']) ? intval($_POST['sort_order']) : 0;
    $status = isset($_POST['status']) ? intval($_POST['status']) : 0;
    $image_type = isset($_POST['image_type']) ? sanitize($_POST['image_type']) : 'current';
    
    // Validasyon
    if (empty($name)) {
        $errors[] = "Kategori adı gereklidir.";
    }
    
    // Resim işleme
    $image_name = $category['image']; // Mevcut resim adını koru
    
    if ($image_type == 'custom' && isset($_FILES['image']) && $_FILES['image']['error'] == 0) {
        // Eski resmi sil (eğer özel yüklenmiş bir resimse)
        if (!empty($category['image']) && !filter_var($category['image'], FILTER_VALIDATE_URL)) {
            $old_image_path = "../uploads/" . $category['image'];
            if (file_exists($old_image_path)) {
                unlink($old_image_path);
            }
        }
        
        $upload_result = uploadImage($_FILES['image']);
        
        if ($upload_result['success']) {
            $image_name = $upload_result['file_name'];
        } else {
            $errors[] = $upload_result['message'];
        }
    } elseif ($image_type == 'unsplash') {
        // Eski resmi sil (eğer özel yüklenmiş bir resimse)
        if (!empty($category['image']) && !filter_var($category['image'], FILTER_VALIDATE_URL)) {
            $old_image_path = "../uploads/" . $category['image'];
            if (file_exists($old_image_path)) {
                unlink($old_image_path);
            }
        }
        
        $image_name = isset($_POST['unsplash_url']) ? sanitize($_POST['unsplash_url']) : '';
    } elseif ($image_type == 'default') {
        // Eski resmi sil (eğer özel yüklenmiş bir resimse)
        if (!empty($category['image']) && !filter_var($category['image'], FILTER_VALIDATE_URL)) {
            $old_image_path = "../uploads/" . $category['image'];
            if (file_exists($old_image_path)) {
                unlink($old_image_path);
            }
        }
        
        $image_name = ''; // Varsayılan resim için boş bırak
    }
    
    // Hata yoksa veritabanını güncelle
    if (empty($errors)) {
        // Slug oluştur
        $slug = slugify($name);
        
        // Veritabanını güncelle
        $stmt = $conn->prepare("UPDATE categories SET name = :name, slug = :slug, description = :description, image = :image, sort_order = :sort_order, status = :status WHERE id = :id");
        $stmt->bindParam(':name', $name);
        $stmt->bindParam(':slug', $slug);
        $stmt->bindParam(':description', $description);
        $stmt->bindParam(':image', $image_name);
        $stmt->bindParam(':sort_order', $sort_order, PDO::PARAM_INT);
        $stmt->bindParam(':status', $status, PDO::PARAM_INT);
        $stmt->bindParam(':id', $id, PDO::PARAM_INT);
        
        if ($stmt->execute()) {
            $success = true;
            
            // Güncel kategori bilgilerini getir
            $stmt = $conn->prepare("SELECT * FROM categories WHERE id = :id");
            $stmt->bindParam(':id', $id, PDO::PARAM_INT);
            $stmt->execute();
            $category = $stmt->fetch();
            
            // Aynı sayfada kal, URL'yi değiştirme
            // header("Location: edit_category.php?id=$id&success=1");
            // exit;
        } else {
            $errors[] = "Kategori güncellenirken bir hata oluştu.";
        }
    }
}

// Kategori resim türünü belirle
$current_image_type = 'default';
if (!empty($category['image'])) {
    if (filter_var($category['image'], FILTER_VALIDATE_URL)) {
        $current_image_type = 'unsplash';
    } else {
        $current_image_type = 'custom';
    }
}
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Kategori Düzenle - <?= $siteTitle; ?></title>
    <meta name="description" content="<?= $category['name']; ?> kategorisini düzenle - <?= $siteTitle; ?> yönetim paneli">
    <?php if (!empty($faviconImage)): ?>
        <?php if (filter_var($faviconImage, FILTER_VALIDATE_URL)): ?>
            <link rel="icon" href="<?= $faviconImage; ?>" type="image/x-icon">
            <link rel="shortcut icon" href="<?= $faviconImage; ?>" type="image/x-icon">
        <?php else: ?>
            <link rel="icon" href="../uploads/<?= $faviconImage; ?>?v=<?= time(); ?>" type="image/x-icon">
            <link rel="shortcut icon" href="../uploads/<?= $faviconImage; ?>?v=<?= time(); ?>" type="image/x-icon">
        <?php endif; ?>
    <?php else: ?>
        <link rel="icon" href="../assets/img/favicon.ico" type="image/x-icon">
        <link rel="shortcut icon" href="../assets/img/favicon.ico" type="image/x-icon">
    <?php endif; ?>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <link rel="stylesheet" href="../assets/css/admin.css">
</head>
<body>
    <?php include 'includes/header.php'; ?>

    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4 py-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">Kategori Düzenle: <?= $category['name']; ?></h1>
                    <div class="btn-toolbar mb-2 mb-md-0">
                        <a href="categories.php" class="btn btn-sm btn-secondary">
                            <i class="fas fa-arrow-left me-1"></i> Kategorilere Dön
                        </a>
                    </div>
                </div>
                
                <?php if ($success): ?>
                     <div class="alert alert-success">
                        Kategori başarıyla güncellendi.
                    </div>
                <?php endif; ?>
                
                <?php if (!empty($errors)): ?>
                    <div class="alert alert-danger">
                        <ul class="mb-0">
                            <?php foreach ($errors as $error): ?>
                                <li><?= $error; ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                <?php endif; ?>
                
                <div class="card">
                    <div class="card-body">
                        <form method="post" action="edit_category.php?id=<?= $id; ?>" enctype="multipart/form-data" class="needs-validation" novalidate>
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label for="name" class="form-label">Kategori Adı <span class="text-danger">*</span></label>
                                        <input type="text" class="form-control" id="name" name="name" value="<?= $category['name']; ?>" required>
                                        <div class="invalid-feedback">Kategori adı gereklidir.</div>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="description" class="form-label">Açıklama</label>
                                        <textarea class="form-control" id="description" name="description" rows="3"><?= $category['description']; ?></textarea>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="sort_order" class="form-label">Sıralama</label>
                                        <input type="number" class="form-control" id="sort_order" name="sort_order" value="<?= $category['sort_order']; ?>">
                                        <div class="form-text">Düşük değerler önce gösterilir.</div>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="status" class="form-label">Durum</label>
                                        <select class="form-select" id="status" name="status">
                                            <option value="1" <?= $category['status'] == 1 ? 'selected' : ''; ?>>Aktif</option>
                                            <option value="0" <?= $category['status'] == 0 ? 'selected' : ''; ?>>Pasif</option>
                                        </select>
                                    </div>
                                </div>
                                
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label class="form-label">Kategori Resmi</label>
                                        
                                        <?php if (!empty($category['image'])): ?>
                                            <div class="mb-3">
                                                <p>Mevcut Resim:</p>
                                                <?php if (filter_var($category['image'], FILTER_VALIDATE_URL)): ?>
                                                    <img src="<?= $category['image']; ?>" alt="<?= $category['name']; ?>" class="img-thumbnail" style="max-width: 200px;">
                                                <?php else: ?>
                                                    <img src="../uploads/<?= $category['image']; ?>?v=<?= time(); ?>" alt="<?= $category['name']; ?>" class="img-thumbnail" style="max-width: 200px;">
                                                <?php endif; ?>
                                            </div>
                                        <?php endif; ?>
                                        
                                        <div class="card">
                                            <div class="card-body">
                                                <div class="mb-3">
                                                    <div class="form-check">
                                                        <input class="form-check-input" type="radio" name="image_type" id="image_type_current" value="current" checked>
                                                        <label class="form-check-label" for="image_type_current">
                                                            Mevcut resmi kullan
                                                        </label>
                                                    </div>
                                                </div>
                                                
                                                <div class="mb-3">
                                                    <div class="form-check">
                                                        <input class="form-check-input" type="radio" name="image_type" id="image_type_default" value="default">
                                                        <label class="form-check-label" for="image_type_default">
                                                            Varsayılan resim kullan
                                                        </label>
                                                    </div>
                                                    <div class="form-text">Sistem kategori adına göre otomatik bir resim seçecektir.</div>
                                                </div>
                                                
                                                <div class="mb-3">
                                                    <div class="form-check">
                                                        <input class="form-check-input" type="radio" name="image_type" id="image_type_unsplash" value="unsplash">
                                                        <label class="form-check-label" for="image_type_unsplash">
                                                            Unsplash resmi kullan
                                                        </label>
                                                    </div>
                                                    <div id="unsplash_url_container" class="mt-2 d-none">
                                                        <input type="text" class="form-control" id="unsplash_url" name="unsplash_url" 
                                                               value="<?= $current_image_type == 'unsplash' ? $category['image'] : '' ?>" 
                                                               placeholder="Unsplash resim URL'si">
                                                        <div class="form-text">Örnek: https://images.unsplash.com/photo-1546241072-48010ad2862c</div>
                                                    </div>
                                                </div>
                                                
                                                <div class="mb-3">
                                                    <div class="form-check">
                                                        <input class="form-check-input" type="radio" name="image_type" id="image_type_custom" value="custom">
                                                        <label class="form-check-label" for="image_type_custom">
                                                            Özel resim yükle
                                                        </label>
                                                    </div>
                                                    <div id="custom_image_container" class="mt-2 d-none">
                                                        <input type="file" class="form-control" id="image" name="image" accept="image/*">
                                                        <div class="form-text">Maksimum dosya boyutu: 2MB. İzin verilen formatlar: JPG, JPEG, PNG, GIF</div>
                                                        <div id="image-preview" class="mt-2"></div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <button type="submit" class="btn btn-primary">Kategoriyi Güncelle</button>
                        </form>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="../assets/js/admin.js"></script>
    <script>
        // Form doğrulama
        (function() {
            'use strict';
            
            // Fetch all the forms we want to apply custom Bootstrap validation styles to
            var forms = document.querySelectorAll('.needs-validation');
            
            // Loop over them and prevent submission
            Array.prototype.slice.call(forms).forEach(function(form) {
                form.addEventListener('submit', function(event) {
                    if (!form.checkValidity()) {
                        event.preventDefault();
                        event.stopPropagation();
                    }
                    
                    form.classList.add('was-validated');
                }, false);
            });
        })();
        
        // Resim seçeneklerini göster/gizle
        document.addEventListener('DOMContentLoaded', function() {
            const imageTypeRadios = document.querySelectorAll('input[name="image_type"]');
            const unsplashUrlContainer = document.getElementById('unsplash_url_container');
            const customImageContainer = document.getElementById('custom_image_container');
            
            // Sayfa yüklendiğinde seçili olan radio butonuna göre konteynerları göster/gizle
            function updateContainers() {
                const selectedValue = document.querySelector('input[name="image_type"]:checked').value;
                
                if (selectedValue === 'unsplash') {
                    unsplashUrlContainer.classList.remove('d-none');
                    customImageContainer.classList.add('d-none');
                } else if (selectedValue === 'custom') {
                    unsplashUrlContainer.classList.add('d-none');
                    customImageContainer.classList.remove('d-none');
                } else {
                    unsplashUrlContainer.classList.add('d-none');
                    customImageContainer.classList.add('d-none');
                }
            }
            
            // İlk yüklemede kontrol et
            updateContainers();
            
            // Radio butonlarına change event listener ekle
            imageTypeRadios.forEach(radio => {
                radio.addEventListener('change', updateContainers);
            });
            
            // Resim önizleme
            const imageInput = document.getElementById('image');
            if (imageInput) {
                imageInput.addEventListener('change', function(e) {
                    const file = e.target.files[0];
                    if (file) {
                        const reader = new FileReader();
                        reader.onload = function(e) {
                            const preview = document.getElementById('image-preview');
                            if (preview) {
                                preview.innerHTML = `<img src="${e.target.result}" class="img-thumbnail" style="max-width: 200px;">`;
                            }
                        }
                        reader.readAsDataURL(file);
                    }
                });
            }
        });
    </script>
</body>
</html>