<?php
session_start();
require_once '../config/db.php';
require_once '../includes/functions.php';

// Cache control headers to prevent browser caching
header("Cache-Control: no-store, no-cache, must-revalidate, max-age=0");
header("Cache-Control: post-check=0, pre-check=0", false);
header("Pragma: no-cache");

// Admin giriş kontrolü
if (!isLoggedIn()) {
    redirectToLogin();
}

// Site ayarlarını getir
$siteSettings = getSiteSettings($conn);
$siteTitle = $siteSettings['site_title'] ?? 'Restoran Menü';
$faviconImage = $siteSettings['favicon_image'] ?? '';

// Ürün ID'sini al
$id = isset($_GET['id']) ? intval($_GET['id']) : 0;

// Ürün bilgilerini getir
$stmt = $conn->prepare("SELECT * FROM products WHERE id = :id");
$stmt->bindParam(':id', $id, PDO::PARAM_INT);
$stmt->execute();
$product = $stmt->fetch();

// Ürün bulunamadıysa ürünler sayfasına yönlendir
if (!$product) {
    header("Location: products.php");
    exit;
}

// Kategorileri getir
$stmt = $conn->prepare("SELECT * FROM categories WHERE status = 1 ORDER BY sort_order ASC");
$stmt->execute();
$categories = $stmt->fetchAll();

$errors = [];
$success = false;

// Form gönderildiğinde
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Form verilerini al
    $name = isset($_POST['name']) ? sanitize($_POST['name']) : '';
    $category_id = isset($_POST['category_id']) ? intval($_POST['category_id']) : 0;
    $description = isset($_POST['description']) ? sanitize($_POST['description']) : '';
    $price = isset($_POST['price']) ? floatval($_POST['price']) : 0;
    $sort_order = isset($_POST['sort_order']) ? intval($_POST['sort_order']) : 0;
    $status = isset($_POST['status']) ? intval($_POST['status']) : 0;
    $featured = isset($_POST['featured']) ? intval($_POST['featured']) : 0;
    
    // Validasyon
    if (empty($name)) {
        $errors[] = "Ürün adı gereklidir.";
    }
    
    if ($category_id <= 0) {
        $errors[] = "Geçerli bir kategori seçmelisiniz.";
    }
    
    if ($price <= 0) {
        $errors[] = "Fiyat sıfırdan büyük olmalıdır.";
    }
    
    // Resim yükleme
    $image_name = $product['image']; // Mevcut resim adını koru
    
    // Resmi kaldır seçeneği işaretlendiyse
    if (isset($_POST['remove_image']) && $_POST['remove_image'] == 1) {
        // Eski resmi sil
        if (!empty($product['image'])) {
            $old_image_path = "../uploads/" . $product['image'];
            if (file_exists($old_image_path)) {
                unlink($old_image_path);
            }
        }
        
        $image_name = '';
    }
    // Yeni resim yüklendiyse
    elseif (isset($_FILES['image']) && $_FILES['image']['error'] == 0) {
        // Eski resmi sil
        if (!empty($product['image'])) {
            $old_image_path = "../uploads/" . $product['image'];
            if (file_exists($old_image_path)) {
                unlink($old_image_path);
            }
        }
        
        $upload_result = uploadImage($_FILES['image']);
        
        if ($upload_result['success']) {
            $image_name = $upload_result['file_name'];
        } else {
            $errors[] = $upload_result['message'];
        }
    }
    
    // Hata yoksa veritabanını güncelle
    if (empty($errors)) {
        // Slug oluştur
        $slug = slugify($name);
        
        // Veritabanını güncelle
        $stmt = $conn->prepare("UPDATE products SET category_id = :category_id, name = :name, slug = :slug, 
                               description = :description, price = :price, image = :image, 
                               sort_order = :sort_order, status = :status, featured = :featured WHERE id = :id");
        $stmt->bindParam(':category_id', $category_id, PDO::PARAM_INT);
        $stmt->bindParam(':name', $name);
        $stmt->bindParam(':slug', $slug);
        $stmt->bindParam(':description', $description);
        $stmt->bindParam(':price', $price);
        $stmt->bindParam(':image', $image_name);
        $stmt->bindParam(':sort_order', $sort_order, PDO::PARAM_INT);
        $stmt->bindParam(':status', $status, PDO::PARAM_INT);
        $stmt->bindParam(':featured', $featured, PDO::PARAM_INT);
        $stmt->bindParam(':id', $id, PDO::PARAM_INT);
        
        if ($stmt->execute()) {
            $success = true;
            
            // Güncel ürün bilgilerini getir
            $stmt = $conn->prepare("SELECT * FROM products WHERE id = :id");
            $stmt->bindParam(':id', $id, PDO::PARAM_INT);
            $stmt->execute();
            $product = $stmt->fetch();
            
            // Aynı sayfada kal, URL'yi değiştirme
            // header("Location: edit_product.php?id=$id&success=1");
            // exit;
        } else {
            $errors[] = "Ürün güncellenirken bir hata oluştu.";
        }
    }
}
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Ürün Düzenle - <?= $siteTitle; ?></title>
    <meta name="description" content="<?= $product['name']; ?> ürününü düzenle - <?= $siteTitle; ?> yönetim paneli">
    <?php if (!empty($faviconImage)): ?>
        <?php if (filter_var($faviconImage, FILTER_VALIDATE_URL)): ?>
            <link rel="icon" href="<?= $faviconImage; ?>" type="image/x-icon">
            <link rel="shortcut icon" href="<?= $faviconImage; ?>" type="image/x-icon">
        <?php else: ?>
            <link rel="icon" href="../uploads/<?= $faviconImage; ?>?v=<?= time(); ?>" type="image/x-icon">
            <link rel="shortcut icon" href="../uploads/<?= $faviconImage; ?>?v=<?= time(); ?>" type="image/x-icon">
        <?php endif; ?>
    <?php else: ?>
        <link rel="icon" href="../assets/img/favicon.ico" type="image/x-icon">
        <link rel="shortcut icon" href="../assets/img/favicon.ico" type="image/x-icon">
    <?php endif; ?>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <link rel="stylesheet" href="../assets/css/admin.css">
</head>
<body>
    <?php include 'includes/header.php'; ?>

    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4 py-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">Ürün Düzenle: <?= $product['name']; ?></h1>
                    <div class="btn-toolbar mb-2 mb-md-0">
                        <a href="products.php" class="btn btn-sm btn-secondary">
                            <i class="fas fa-arrow-left me-1"></i> Ürünlere Dön
                        </a>
                    </div>
                </div>
                
                <?php if ($success): ?>
                    <div class="alert alert-success">
                        Ürün başarıyla güncellendi.
                    </div>
                <?php endif; ?>
                
                <?php if (!empty($errors)): ?>
                    <div class="alert alert-danger">
                        <ul class="mb-0">
                            <?php foreach ($errors as $error): ?>
                                <li><?= $error; ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                <?php endif; ?>
                
                <div class="card">
                    <div class="card-body">
                        <form method="post" action="edit_product.php?id=<?= $id; ?>" enctype="multipart/form-data" class="needs-validation" novalidate>
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label for="name" class="form-label">Ürün Adı <span class="text-danger">*</span></label>
                                        <input type="text" class="form-control" id="name" name="name" value="<?= $product['name']; ?>" required>
                                        <div class="invalid-feedback">Ürün adı gereklidir.</div>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="category_id" class="form-label">Kategori <span class="text-danger">*</span></label>
                                        <select class="form-select" id="category_id" name="category_id" required>
                                            <option value="">Kategori Seçin</option>
                                            <?php foreach ($categories as $category): ?>
                                                <option value="<?= $category['id']; ?>" <?= $category['id'] == $product['category_id'] ? 'selected' : ''; ?>>
                                                    <?= $category['name']; ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                        <div class="invalid-feedback">Geçerli bir kategori seçmelisiniz.</div>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="price" class="form-label">Fiyat (₺) <span class="text-danger">*</span></label>
                                        <input type="number" class="form-control" id="price" name="price" step="0.01" min="0.01" value="<?= $product['price']; ?>" required>
                                        <div class="invalid-feedback">Fiyat sıfırdan büyük olmalıdır.</div>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="image" class="form-label">Ürün Resmi</label>
                                        <?php if (!empty($product['image'])): ?>
                                            <div class="mb-2">
                                                <img src="../uploads/<?= $product['image']; ?>?v=<?= time(); ?>" alt="<?= $product['name']; ?>" class="img-thumbnail" style="max-width: 200px;">
                                                <div class="form-check mt-2">
                                                    <input class="form-check-input" type="checkbox" id="remove_image" name="remove_image" value="1">
                                                    <label class="form-check-label" for="remove_image">
                                                        Resmi kaldır
                                                    </label>
                                                </div>
                                            </div>
                                        <?php endif; ?>
                                        <input type="file" class="form-control" id="image" name="image" accept="image/*">
                                        <div class="form-text">Maksimum dosya boyutu: 2MB. İzin verilen formatlar: JPG, JPEG, PNG, GIF</div>
                                        <div id="image-preview" class="mt-2"></div>
                                    </div>
                                </div>
                                
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label for="description" class="form-label">Açıklama</label>
                                        <textarea class="form-control" id="description" name="description" rows="5"><?= $product['description']; ?></textarea>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="sort_order" class="form-label">Sıralama</label>
                                        <input type="number" class="form-control" id="sort_order" name="sort_order" value="<?= $product['sort_order']; ?>">
                                        <div class="form-text">Düşük değerler önce gösterilir.</div>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="status" class="form-label">Durum</label>
                                        <select class="form-select" id="status" name="status">
                                            <option value="1" <?= $product['status'] == 1 ? 'selected' : ''; ?>>Aktif</option>
                                            <option value="0" <?= $product['status'] == 0 ? 'selected' : ''; ?>>Pasif</option>
                                        </select>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="featured" class="form-label">Öne Çıkan</label>
                                        <select class="form-select" id="featured" name="featured">
                                            <option value="1" <?= $product['featured'] == 1 ? 'selected' : ''; ?>>Evet</option>
                                            <option value="0" <?= $product['featured'] == 0 ? 'selected' : ''; ?>>Hayır</option>
                                        </select>
                                        <div class="form-text">Öne çıkan ürünler ana sayfada gösterilir.</div>
                                    </div>
                                </div>
                            </div>
                            
                            <button type="submit" class="btn btn-primary">Ürünü Güncelle</button>
                        </form>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="../assets/js/admin.js"></script>
    <script>
        // Form doğrulama
        (function() {
            'use strict';
            
            // Fetch all the forms we want to apply custom Bootstrap validation styles to
            var forms = document.querySelectorAll('.needs-validation');
            
            // Loop over them and prevent submission
            Array.prototype.slice.call(forms).forEach(function(form) {
                form.addEventListener('submit', function(event) {
                    if (!form.checkValidity()) {
                        event.preventDefault();
                        event.stopPropagation();
                    }
                    
                    form.classList.add('was-validated');
                }, false);
            });
        })();
        
        // Resmi kaldır checkbox'ı için
        document.addEventListener('DOMContentLoaded', function() {
            const removeImageCheckbox = document.getElementById('remove_image');
            if (removeImageCheckbox) {
                removeImageCheckbox.addEventListener('change', function() {
                    const imageInput = document.getElementById('image');
                    const currentImage = this.parentElement.parentElement.querySelector('img');
                    
                    if (this.checked) {
                        if (currentImage) {
                            currentImage.style.opacity = '0.3';
                        }
                        if (imageInput) {
                            imageInput.disabled = true;
                        }
                    } else {
                        if (currentImage) {
                            currentImage.style.opacity = '1';
                        }
                        if (imageInput) {
                            imageInput.disabled = false;
                        }
                    }
                });
            }
            
            // Resim önizleme
            const imageInput = document.getElementById('image');
            if (imageInput) {
                imageInput.addEventListener('change', function(e) {
                    const file = e.target.files[0];
                    if (file) {
                        const reader = new FileReader();
                        reader.onload = function(e) {
                            const preview = document.getElementById('image-preview');
                            if (preview) {
                                preview.innerHTML = `<img src="${e.target.result}" class="img-thumbnail" style="max-width: 200px;">`;
                            }
                        }
                        reader.readAsDataURL(file);
                    }
                });
            }
        });
    </script>
</body>
</html>