<?php
session_start();
require_once '../config/db.php';
require_once '../includes/functions.php';

// Güvenlik için HTTP başlıkları
header("X-XSS-Protection: 1; mode=block");
header("X-Content-Type-Options: nosniff");
header("X-Frame-Options: DENY");
header("Content-Security-Policy: default-src 'self'; script-src 'self' https://cdn.jsdelivr.net https://code.jquery.com; style-src 'self' https://cdn.jsdelivr.net https://cdnjs.cloudflare.com 'unsafe-inline'; img-src 'self' data: https:; font-src 'self' https://cdnjs.cloudflare.com; connect-src 'self';");

// Kullanıcı zaten giriş yapmışsa yönetim paneline yönlendir
if (isLoggedIn()) {
    header("Location: index.php");
    exit;
}

// CSRF token oluştur
$csrf_token = generateCSRFToken();

$error = '';

// Form gönderildiğinde
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // CSRF token kontrolü
    if (!isset($_POST['csrf_token']) || !validateCSRFToken($_POST['csrf_token'])) {
        $error = 'Güvenlik doğrulaması başarısız oldu. Lütfen sayfayı yenileyip tekrar deneyin.';
    } else {
        $username = isset($_POST['username']) ? sanitize($_POST['username']) : '';
        $password = isset($_POST['password']) ? $_POST['password'] : '';
        
        // Kullanıcı adı ve şifre kontrolü
        if (empty($username) || empty($password)) {
            $error = 'Kullanıcı adı ve şifre gereklidir.';
        } else {
            // Brute force saldırılarını önlemek için giriş denemelerini sınırla
            if (isset($_SESSION['login_attempts']) && $_SESSION['login_attempts'] >= 5) {
                if (time() - $_SESSION['last_attempt'] < 300) { // 5 dakika bekletme
                    $error = 'Çok fazla başarısız giriş denemesi. Lütfen 5 dakika sonra tekrar deneyin.';
                    header("Retry-After: 300");
                } else {
                    // Süre dolmuşsa sayacı sıfırla
                    $_SESSION['login_attempts'] = 0;
                }
            }
            
            if (empty($error)) {
                // Veritabanından kullanıcıyı kontrol et
                try {
                    $stmt = $conn->prepare("SELECT * FROM users WHERE username = :username AND status = 1");
                    $stmt->bindParam(':username', $username);
                    $stmt->execute();
                    $user = $stmt->fetch();
                    
                    if ($user && password_verify($password, $user['password'])) {
                        // Giriş başarılı
                        loginSuccess($conn, $user);
                    } else {
                        // Başarısız giriş
                        loginFailed($conn, $username);
                        $error = 'Geçersiz kullanıcı adı veya şifre.';
                    }
                } catch (PDOException $e) {
                    error_log("Kullanıcı giriş hatası: " . $e->getMessage());
                    $error = 'Giriş işlemi sırasında bir hata oluştu. Lütfen daha sonra tekrar deneyin.';
                }
            }
        }
    }
}

// Başarılı giriş işlemi
function loginSuccess($conn, $user) {
    $_SESSION['admin_logged_in'] = true;
    $_SESSION['admin_id'] = $user['id'];
    $_SESSION['admin_username'] = $user['username'];
    $_SESSION['admin_role'] = $user['role'];
    
    // Giriş denemelerini sıfırla
    unset($_SESSION['login_attempts']);
    unset($_SESSION['last_attempt']);
    
    // Oturum ID'sini yenile (session fixation koruması)
    session_regenerate_id(true);
    
    // Giriş logunu kaydet
    $ip_address = $_SERVER['REMOTE_ADDR'];
    $user_agent = $_SERVER['HTTP_USER_AGENT'];
    $login_time = date('Y-m-d H:i:s');
    $status = 'success';
    
    $stmt = $conn->prepare("INSERT INTO login_logs (user_id, username, ip_address, user_agent, login_time, status) 
                         VALUES (:user_id, :username, :ip_address, :user_agent, :login_time, :status)");
    $stmt->bindParam(':user_id', $user['id']);
    $stmt->bindParam(':username', $user['username']);
    $stmt->bindParam(':ip_address', $ip_address);
    $stmt->bindParam(':user_agent', $user_agent);
    $stmt->bindParam(':login_time', $login_time);
    $stmt->bindParam(':status', $status);
    $stmt->execute();
    
    // Yönetim paneline yönlendir
    header("Location: index.php");
    exit;
}

// Başarısız giriş işlemi
function loginFailed($conn, $username) {
    // Başarısız giriş denemelerini takip et
    if (!isset($_SESSION['login_attempts'])) {
        $_SESSION['login_attempts'] = 1;
    } else {
        $_SESSION['login_attempts']++;
    }
    $_SESSION['last_attempt'] = time();
    
    // Başarısız giriş logunu kaydet
    $ip_address = $_SERVER['REMOTE_ADDR'];
    $user_agent = $_SERVER['HTTP_USER_AGENT'];
    $login_time = date('Y-m-d H:i:s');
    $status = 'failed';
    
    $stmt = $conn->prepare("INSERT INTO login_logs (username, ip_address, user_agent, login_time, status) 
                         VALUES (:username, :ip_address, :user_agent, :login_time, :status)");
    $stmt->bindParam(':username', $username);
    $stmt->bindParam(':ip_address', $ip_address);
    $stmt->bindParam(':user_agent', $user_agent);
    $stmt->bindParam(':login_time', $login_time);
    $stmt->bindParam(':status', $status);
    $stmt->execute();
}

// Site ayarlarını getir
$siteSettings = getSiteSettings($conn);
$siteTitle = $siteSettings['site_title'] ?? 'Restoran Menü';
$faviconImage = $siteSettings['favicon_image'] ?? '';
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Yönetici Girişi - <?= safeOutput($siteTitle); ?></title>
    <?php if (!empty($faviconImage)): ?>
        <?php if (filter_var($faviconImage, FILTER_VALIDATE_URL)): ?>
            <link rel="icon" href="<?= safeOutput($faviconImage); ?>" type="image/x-icon">
            <link rel="shortcut icon" href="<?= safeOutput($faviconImage); ?>" type="image/x-icon">
        <?php else: ?>
            <link rel="icon" href="../uploads/<?= safeOutput($faviconImage); ?>?v=<?= time(); ?>" type="image/x-icon">
            <link rel="shortcut icon" href="../uploads/<?= safeOutput($faviconImage); ?>?v=<?= time(); ?>" type="image/x-icon">
        <?php endif; ?>
    <?php else: ?>
        <link rel="icon" href="../assets/img/favicon.ico" type="image/x-icon">
        <link rel="shortcut icon" href="../assets/img/favicon.ico" type="image/x-icon">
    <?php endif; ?>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        body {
            background-color: #f8f9fa;
            height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        .login-form {
            max-width: 400px;
            width: 100%;
            padding: 30px;
            background-color: #fff;
            border-radius: 10px;
            box-shadow:  0 0 20px rgba(0, 0, 0, 0.1);
        }
        .login-logo {
            text-align: center;
            margin-bottom: 30px;
        }
        .login-logo i {
            font-size: 50px;
            color: #0d6efd;
        }
    </style>
</head>
<body>
    <div class="login-form">
        <div class="login-logo">
            <i class="fas fa-utensils"></i>
            <h2><?= safeOutput($siteTitle); ?></h2>
            <p class="text-muted">Yönetici Girişi</p>
        </div>
        
        <?php if (!empty($error)): ?>
            <div class="alert alert-danger"><?= safeOutput($error); ?></div>
        <?php endif; ?>
        
        <form method="post" action="" class="needs-validation" novalidate>
            <input type="hidden" name="csrf_token" value="<?= $csrf_token; ?>">
            
            <div class="mb-3">
                <label for="username" class="form-label">Kullanıcı Adı</label>
                <div class="input-group">
                    <span class="input-group-text"><i class="fas fa-user"></i></span>
                    <input type="text" class="form-control" id="username" name="username" required autocomplete="username">
                    <div class="invalid-feedback">Kullanıcı adı gereklidir.</div>
                </div>
            </div>
            <div class="mb-4">
                <label for="password" class="form-label">Şifre</label>
                <div class="input-group">
                    <span class="input-group-text"><i class="fas fa-lock"></i></span>
                    <input type="password" class="form-control" id="password" name="password" required autocomplete="current-password">
                    <div class="invalid-feedback">Şifre gereklidir.</div>
                </div>
            </div>
            <div class="d-grid">
                <button type="submit" class="btn btn-primary">Giriş Yap</button>
            </div>
        </form>
        
        <div class="mt-4 text-center">
            <a href="../index.php" class="text-decoration-none">
                <i class="fas fa-arrow-left me-1"></i> Ana Sayfaya Dön
            </a>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Form doğrulama
        (function() {
            'use strict';
            
            var forms = document.querySelectorAll('.needs-validation');
            
            Array.prototype.slice.call(forms).forEach(function(form) {
                form.addEventListener('submit', function(event) {
                    if (!form.checkValidity()) {
                        event.preventDefault();
                        event.stopPropagation();
                    }
                    
                    form.classList.add('was-validated');
                }, false);
            });
        })();
    </script>
</body>
</html>