<?php
session_start();
require_once 'config/db.php';
require_once 'includes/functions.php';

// Güvenlik için HTTP başlıkları
header("X-XSS-Protection: 1; mode=block");
header("X-Content-Type-Options: nosniff");
header("X-Frame-Options: SAMEORIGIN");
header("Referrer-Policy: strict-origin-when-cross-origin");
header("Content-Security-Policy: default-src 'self'; script-src 'self' https://cdn.jsdelivr.net https://code.jquery.com; style-src 'self' https://cdn.jsdelivr.net https://cdnjs.cloudflare.com 'unsafe-inline'; img-src 'self' data: https:; font-src 'self' https://cdnjs.cloudflare.com; connect-src 'self';");

// Cache control headers to prevent browser caching
header("Cache-Control: no-store, no-cache, must-revalidate, max-age=0");
header("Cache-Control: post-check=0, pre-check=0", false);
header("Pragma: no-cache");

// URL'den kategori ID ve slug'ı al
$category_id = 0;
$slug = '';

// SEO dostu URL yapısı için
$request_uri = $_SERVER['REQUEST_URI'];
if (preg_match('/kategori\/(\d+)-([a-z0-9\-]+)\.html/', $request_uri, $matches)) {
    $category_id = intval($matches[1]);
    $slug = $matches[2];
} else {
    // Eski URL yapısı için geriye dönük uyumluluk
    $category_id = isset($_GET['id']) ? intval($_GET['id']) : 0;
    $slug = isset($_GET['slug']) ? $_GET['slug'] : '';
}

// Kategori bilgilerini getir
$category = getCategoryById($conn, $category_id);

// Kategori bulunamadıysa ana sayfaya yönlendir
if (!$category) {
    header("Location: index.php");
    exit;
}

// URL'deki slug ile kategori slug'ı uyuşmuyorsa doğru URL'ye yönlendir
if ($slug != $category['slug']) {
    header("Location: kategori/{$category_id}-{$category['slug']}.html");
    exit;
}

// Kategoriye ait ürünleri getir
$products = getProductsByCategory($conn, $category_id);

// Tüm kategorileri getir (menü için)
$categories = getCategories($conn);

// Site ayarlarını getir
$siteSettings = getSiteSettings($conn);
$siteTitle = $siteSettings['site_title'] ?? 'Restoran Menü';
$siteDescription = $siteSettings['site_description'] ?? 'Lezzetli yemeklerimizi keşfedin';
$logoImage = $siteSettings['logo_image'] ?? '';
$faviconImage = $siteSettings['favicon_image'] ?? '';
$headerColor = $siteSettings['header_color'] ?? '#212529';
$footerColor = $siteSettings['footer_color'] ?? '#212529';
$showHeader = $siteSettings['show_header'] ?? '1';
$showFooter = $siteSettings['show_footer'] ?? '1';

// Kategori resmi için URL belirle
if (!empty($category['image'])) {
    if (filter_var($category['image'], FILTER_VALIDATE_URL)) {
        // Unsplash veya harici URL
        $categoryImgUrl = $category['image'];
    } else {
        // Yüklenmiş resim
        $categoryImgUrl = "../uploads/" . $category['image'] . "?v=" . time(); // Cache busting
    }
} else {
    // Varsayılan resim
    $categoryImgUrl = getCategoryDefaultImage($category['name']);
}
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= safeOutput($category['name']); ?> - <?= safeOutput($siteTitle); ?></title>
    <meta name="description" content="<?= safeOutput(strip_tags($category['description'])); ?>">
    <meta name="keywords" content="<?= safeOutput($category['name']); ?>, menü, restoran, yemek, <?= safeOutput($siteTitle); ?>">
    <meta property="og:title" content="<?= safeOutput($category['name']); ?> - <?= safeOutput($siteTitle); ?>">
    <meta property="og:description" content="<?= safeOutput(strip_tags($category['description'])); ?>">
    <meta property="og:image" content="<?= safeOutput($categoryImgUrl); ?>">
    <meta property="og:url" content="<?= "https://" . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI']; ?>">
    <meta property="og:type" content="website">
    <?php if (!empty($faviconImage)): ?>
        <?php if (filter_var($faviconImage, FILTER_VALIDATE_URL)): ?>
            <link rel="icon" href="<?= safeOutput($faviconImage); ?>" type="image/x-icon">
            <link rel="shortcut icon" href="<?= safeOutput($faviconImage); ?>" type="image/x-icon">
        <?php else: ?>
            <link rel="icon" href="../uploads/<?= safeOutput($faviconImage); ?>?v=<?= time(); ?>" type="image/x-icon">
            <link rel="shortcut icon" href="../uploads/<?= safeOutput($faviconImage); ?>?v=<?= time(); ?>" type="image/x-icon">
        <?php endif; ?>
    <?php else: ?>
        <link rel="icon" href="../assets/img/favicon.ico" type="image/x-icon">
        <link rel="shortcut icon" href="../assets/img/favicon.ico" type="image/x-icon">
    <?php endif; ?>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <link rel="stylesheet" href="../assets/css/style.css">
    <style>
        .header-custom {
            background-color: <?= $headerColor; ?>;
        }
        .footer-custom {
            background-color: <?= $footerColor; ?>;
        }
        .product-card .card-img-top {
            height: 180px;
            object-fit: cover;
        }
        .product-card .card-body {
            display: flex;
            flex-direction: column;
        }
        .product-card .card-title {
            margin-bottom: 0.5rem;
        }
        .product-card .card-text {
            flex-grow: 1;
            overflow: hidden;
            text-overflow: ellipsis;
            display: -webkit-box;
            -webkit-line-clamp: 3;
            -webkit-box-orient: vertical;
        }
        .product-card .price {
            margin-top: auto;
            font-weight: bold;
            font-size: 1.2rem;
        }
        .modal-body img {
            max-width: 100%;
            max-height: 300px;
            object-fit: cover;
            margin-bottom: 15px;
        }
        /* Fix for logo and favicon paths */
        .logo-img {
            max-height: 50px;
            max-width: 200px;
            width: auto;
        }
    </style>
</head>
<body>
    <?php if ($showHeader == '1'): ?>
    <!-- Basit Logo Header -->
    <div class="text-center py-3 header-custom">
        <div class="container">
            <div class="row align-items-center">
                <div class="col-2 text-start">
                    <a href="../index.php" class="btn btn-outline-light btn-sm">
                        <i class="fas fa-arrow-left"></i> Geri
                    </a>
                </div>
                <div class="col-8">
                    <?php if (!empty($logoImage)): ?>
                        <?php if (filter_var($logoImage, FILTER_VALIDATE_URL)): ?>
                            <img src="<?= safeOutput($logoImage); ?>" alt="<?= safeOutput($siteTitle); ?>" class="logo-img">
                        <?php else: ?>
                            <img src="../uploads/<?= safeOutput($logoImage); ?>?v=<?= time(); ?>" alt="<?= safeOutput($siteTitle); ?>" class="logo-img">
                        <?php endif; ?>
                    <?php else: ?>
                        <h1 class="text-white fs-3"><i class="fas fa-utensils me-2"></i><?= safeOutput($siteTitle); ?></h1>
                    <?php endif; ?>
                </div>
                <div class="col-2"></div>
            </div>
        </div>
    </div>
    <?php endif; ?>

    <!-- Kategori Banner -->
    <div class="category-banner" style="background-image: url('<?= safeOutput($categoryImgUrl); ?>')">
        <div class="container">
            <div class="category-banner-content">
                <h1><?= safeOutput($category['name']); ?></h1>
                <?php if(!empty($category['description'])): ?>
                    <p class="lead"><?= safeOutput($category['description']); ?></p>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <div class="container mt-4">
        <div class="row justify-content-center">
            <?php if(count($products) > 0): ?>
                <?php foreach($products as $product): ?>
                <div class="col-6 col-md-4 col-lg-3 mb-4 product-item">
                    <div class="card product-card h-100">
                        <?php if(!empty($product['image'])): ?>
                            <img src="../uploads/<?= safeOutput($product['image']); ?>?v=<?= time(); ?>" class="card-img-top" alt="<?= safeOutput($product['name']); ?>" loading="lazy">
                        <?php else: ?>
                            <img src="https://images.unsplash.com/photo-1546069901-ba9599a7e63c?q=80&w=500" class="card-img-top" alt="<?= safeOutput($product['name']); ?>" loading="lazy">
                        <?php endif; ?>
                        <div class="card-body">
                            <h5 class="card-title"><?= safeOutput($product['name']); ?></h5>
                            <?php if(!empty($product['description'])): ?>
                                <p class="card-text product-description" data-product-id="<?= $product['id']; ?>"><?= safeOutput(substr($product['description'], 0, 80) . (strlen($product['description']) > 80 ? '...' : '')); ?></p>
                                <?php if(strlen($product['description']) > 80): ?>
                                    <button class="btn btn-sm btn-link p-0 mb-2 read-more" data-bs-toggle="modal" data-bs-target="#productModal<?= $product['id']; ?>">Devamını oku</button>
                                <?php endif; ?>
                            <?php endif; ?>
                            <p class="price"><?= number_format($product['price'], 2); ?> ₺</p>
                        </div>
                    </div>
                </div>

                <!-- Ürün Detay Modal -->
                <?php if(!empty($product['description']) && strlen($product['description']) > 80): ?>
                <div class="modal fade" id="productModal<?= $product['id']; ?>" tabindex="-1" aria-labelledby="productModalLabel<?= $product['id']; ?>" aria-hidden="true">
                    <div class="modal-dialog modal-dialog-centered">
                        <div class="modal-content">
                            <div class="modal-header">
                                <h5 class="modal-title" id="productModalLabel<?= $product['id']; ?>"><?= safeOutput($product['name']); ?></h5>
                                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                            </div>
                            <div class="modal-body">
                                <?php if(!empty($product['image'])): ?>
                                    <img src="../uploads/<?= safeOutput($product['image']); ?>?v=<?= time(); ?>" class="img-fluid rounded" alt="<?= safeOutput($product['name']); ?>">
                                <?php endif; ?>
                                <p><?= safeOutput($product['description']); ?></p>
                                <p class="price"><?= number_format($product['price'], 2); ?> ₺</p>
                            </div>
                            <div class="modal-footer">
                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Kapat</button>
                            </div>
                        </div>
                    </div>
                </div>
                <?php endif; ?>
                <?php endforeach; ?>
            <?php else: ?>
                <div class="col-12 col-md-8">
                    <div class="alert alert-warning">
                        Bu kategoride henüz ürün bulunmamaktadır.
                    </div>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <?php if ($showFooter == '1'): ?>
    <?php include '../includes/footer.php'; ?>
    <?php endif; ?>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="../assets/js/main.js"></script>
</body>
</html>